# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from playwright.sync_api import Page, expect

from e2e_playwright.conftest import ImageCompareFunction
from e2e_playwright.shared.app_utils import check_top_level_class, get_element_by_key


def test_st_json_displays_correctly(app: Page, assert_snapshot: ImageCompareFunction):
    """Test st.json renders the data correctly."""
    json_elements = app.get_by_test_id("stJson")
    expect(json_elements).to_have_count(10)

    assert_snapshot(json_elements.nth(0), name="st_json-simple_dict")
    assert_snapshot(json_elements.nth(1), name="st_json-collapsed")
    assert_snapshot(json_elements.nth(2), name="st_json-with_white_spaces")
    # The complex dict is screenshot tested in the themed test below
    assert_snapshot(json_elements.nth(4), name="st_json-simple_list")
    assert_snapshot(json_elements.nth(5), name="st_json-empty_dict")
    assert_snapshot(json_elements.nth(6), name="st_json-expanded_2")
    # Width tests
    assert_snapshot(json_elements.nth(7), name="st_json-width_stretch")
    assert_snapshot(json_elements.nth(8), name="st_json-width_pixels")
    # The container bounds test is screenshot tested in another test below


def test_st_json_keeps_container_bounds(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Test st.json keeps the container bounds."""
    container_with_json = get_element_by_key(app, "container_with_json")
    expect(container_with_json.get_by_test_id("stJson")).to_have_count(1)
    assert_snapshot(container_with_json, name="st_json-keep_bounds")


def test_st_json_displays_correctly_when_themed(
    themed_app: Page, assert_snapshot: ImageCompareFunction
):
    """Test st.json uses renders the data correctly with different themes."""
    json_elements = themed_app.get_by_test_id("stJson")
    assert_snapshot(json_elements.nth(3), name="st_json-complex_dict")


def test_check_top_level_class(app: Page):
    """Check that the top level class is correctly set."""
    check_top_level_class(app, "stJson")


def test_shows_copy_icon(themed_app: Page, assert_snapshot: ImageCompareFunction):
    """Test that the copy icon is shown by hovering over the element."""
    json_element = themed_app.get_by_test_id("stJson").first
    expect(json_element).to_be_visible()
    json_element.hover()

    assert_snapshot(json_element, name="st_json-copy_icon_on_hover")


def test_shows_json_path_tooltip_on_click(
    themed_app: Page, assert_snapshot: ImageCompareFunction
):
    """Test that clicking on a JSON value shows a tooltip with the JSON path."""
    # Use a JSON element with nested values:
    json_element = themed_app.get_by_test_id("stJson").nth(6)
    expect(json_element).to_be_visible()

    # Click on a string value to trigger the tooltip
    string_value = json_element.locator(".string-value").first
    string_value.click()

    # Wait for and verify the tooltip appears
    tooltip = themed_app.get_by_test_id("stJsonPathTooltip")
    expect(tooltip).to_be_visible()

    assert_snapshot(tooltip, name="st_json-path_tooltip")
