# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import pandas as pd

import streamlit as st
from streamlit import runtime

options = ("female", "male")
markdown_options = (
    "**bold text**",
    "*italics text*",
    "~strikethrough text~",
    "shortcode: :blush:",
    # link should not work in radio options
    "[link text](www.example.com)",
    "`code text`",
    ":red[red] :blue[blue] :green[green] :violet[violet] :orange[orange]",
)

v1 = st.radio("radio 1 (default)", options)
st.write("value 1:", v1)

v2 = st.radio(
    "radio 2 (Formatted options)",
    options,
    1,
    format_func=lambda x: x.capitalize(),
)
st.write("value 2:", v2)

v3 = st.radio("radio 3 (no options)", [])
st.write("value 3:", v3)

v4 = st.radio("radio 4 (disabled)", options, disabled=True)
st.write("value 4:", v4)

v5 = st.radio("radio 5 (horizontal)", options, horizontal=True)
st.write("value 5:", v5)

v6: str = st.radio(
    "radio 6 (options from dataframe)", pd.DataFrame({"foo": list(options)})
)
st.write("value 6:", v6)

v7 = st.radio(
    "radio 7 (hidden label)", options, label_visibility="hidden", key="radio_7"
)
st.write("value 7:", v7)

v8 = st.radio(
    "radio 8 (collapsed label)", options, label_visibility="collapsed", key="radio_8"
)
st.write("value 8:", v8)

v9 = st.radio("radio 9 (markdown options)", options=markdown_options)
st.write("value 9:", v9)

v10 = st.radio(
    "radio 10 (with captions)",
    ["A", "B", "C", "D", "E", "F", "G"],
    captions=markdown_options,
)
st.write("value 10:", v10)

v11 = st.radio(
    "radio 11 (horizontal, captions)",
    ["yes", "maybe", "no"],
    captions=["Opt in", "", "Opt out"],
    horizontal=True,
)
st.write("value 11:", v11)

if runtime.exists():

    def on_change():
        st.session_state.radio_changed = True
        st.text("Radio widget callback triggered")

    st.radio(
        "radio 12 (with callback, help)",
        options,
        1,
        key="radio12",
        on_change=on_change,
        help="help text",
    )
    st.write("value 12:", st.session_state.radio12)
    st.write("radio changed:", st.session_state.get("radio_changed") is True)
    # Reset to False:
    st.session_state.radio_changed = False

v13 = st.radio("radio 13 (empty selection)", options, index=None)
st.write("value 13:", v13)

st.radio(
    "radio 14 -> :material/check: :rainbow[Fancy] _**markdown** `label` _support_",
    options=markdown_options,
)

st.header("Radio - width examples")

st.radio(
    "Radio with content width (default)",
    ["Option 1", "Option 2", "Option 3"],
    width="content",
)

st.radio(
    "Radio with stretch width",
    ["Option A", "Option B", "Option C"],
    width="stretch",
)

st.radio(
    "Radio with 200px width. Label is too long to fit in the width",
    ["Choice 1", "Choice 2", "Choice 3"],
    width=200,
)

if st.toggle("Update radio props"):
    dr_value = st.radio(
        "Updated dynamic radio",
        key="dynamic_radio_with_key",
        help="updated help",
        width=300,
        horizontal=True,
        on_change=lambda a, param: print(
            f"Updated radio - callback triggered: {a} {param}"
        ),
        args=("Updated radio arg",),
        kwargs={"param": "updated kwarg param"},
        captions=["🍎", "🍌", "🍊"],
        # Whitelisted kwargs:
        options=["apple", "banana", "orange"],
        format_func=lambda x: x.capitalize(),
    )
    st.write("Updated radio value:", dr_value)
else:
    dr_value = st.radio(
        "Initial dynamic radio",
        key="dynamic_radio_with_key",
        help="initial help",
        width="content",
        horizontal=False,
        on_change=lambda a, param: print(
            f"Initial radio - callback triggered: {a} {param}"
        ),
        args=("Initial radio arg",),
        kwargs={"param": "initial kwarg param"},
        captions=["🍎 Apple", "🍌 Banana", "🍊 Orange"],
        # Whitelisted kwargs:
        options=["apple", "banana", "orange"],
        format_func=lambda x: x.capitalize(),
    )
    st.write("Initial radio value:", dr_value)
