# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import time

import streamlit as st

with st.sidebar:
    st.markdown(
        """
        - [Multi Select - Segmented Control](#multi-select-segmented-control)
        - [Single Select - Segmented Control](#single-select-segmented-control)
        - [Icon-only button group - Segmented Control](#icon-only-button-group-segmented-control)
        - [on_change callback - Segmented Control](#on-change-callback-segmented-control)
        - [Disabled - Segmented Control](#disabled-segmented-control)
        - [Segmented Control in form](#segmented-control-in-form)
        - [Segmented Control in fragment](#segmented-control-in-fragment)
        - [Unmounted - Segmented Control](#unmounted-segmented-control)
        """
    )

st.header("Multi Select - Segmented Control", anchor="multi-select-segmented-control")


s1 = st.segmented_control(
    "Select some options",
    [
        ":material/star: Hello there!",
        "Foobar",
        "Icon in the end: :material/rocket:",
        ":material/thumb_up: Hello again!",
        "🧰 General widgets",
        "📊 Charts",
        "🌇 Images",
        "🎥 Video",
        "📝 Text",
        (
            "This is a very long text 📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝📝"
            ", yes, long long long long text"
            ", yes, long long long long text"
            ", yes, long long long long text"
        ),
    ],
    key="segmented_control_multi_selection",
    selection_mode="multi",
    help="You can choose multiple options",
)
st.write(f"Multi selection: {s1}")


st.header("Single Select - Segmented Control", anchor="single-select-segmented-control")
s2 = st.segmented_control(
    "Select an option",
    [
        ":material/star: Hello there!",
        "Foobar",
        "Icon in the end: :material/rocket:",
    ],
    key="segmented_control_single_selection",
    selection_mode="single",
)
st.write(f"Single selection: {s2}")

option_to_icon_map = {
    0: ":material/add:",
    1: ":material/zoom_in:",
    2: ":material/zoom_out:",
    3: ":material/zoom_out_map:",
}

st.header(
    "Icon-only button group - Segmented Control",
    anchor="icon-only-button-group-segmented-control",
)
s3 = st.segmented_control(
    "select an icon",
    options=[0, 1, 2, 3],
    format_func=lambda option: option_to_icon_map[option],
    key="segmented_control_single_icon_selection",
    selection_mode="single",
)
st.write(f"Single icon selection: {s3}")


st.header(
    "on_change callback - Segmented Control",
    anchor="on-change-callback-segmented-control",
)
st.segmented_control(
    "Select an emotion:",
    ["Joy", "Sadness", "Anger", "Disgust"],
    key="segmented_control_on_change",
    on_change=lambda: st.write(
        f"on_change selection: {st.session_state.segmented_control_on_change}"
    ),
)


st.header(
    "Disabled - Segmented Control (label collapsed)",
    anchor="disabled-segmented-control",
)
s4 = st.segmented_control(
    "Select an emotion:",
    ["Joy", "Sadness", "Anger", "Disgust"],
    key="segmented_control_disabled",
    disabled=True,
    label_visibility="collapsed",
)
st.write("segmented-control-disabled:", str(s4))


st.header("Segmented Control in form", anchor="segmented-control-in-form")
with st.form(key="my_form", clear_on_submit=True):
    st.segmented_control(
        "Select an emotion:",
        ["Joy", "Sadness", "Anger", "Disgust"],
        key="segmented_control_in_form",
        selection_mode="multi",
    )
    st.form_submit_button("Submit")
st.write(
    "segmented-control-in-form:",
    str(st.session_state.segmented_control_in_form)
    if "segmented_control_in_form" in st.session_state
    else None,
)


st.header("Segmented Control in fragment", anchor="segmented-control-in-fragment")


@st.fragment
def test_fragment():
    s5 = st.segmented_control(
        "Select an emotion:",
        ["Joy", "Sadness", "Anger", "Disgust"],
        key="segmented_control_in_fragment",
    )
    st.write("segmented-control-in-fragment:", str(s5))


test_fragment()


st.header("Unmounted - Segmented Control", anchor="unmounted-segmented-control")
if st.button("Create some elements to unmount component"):
    for _ in range(2):
        # The sleep here is needed, because it won't unmount the
        # component if this is too fast.
        time.sleep(1)
        st.write("Another element")

s6 = st.segmented_control(
    "Select an emotion:",
    ["Joy", "Sadness", "Anger", "Disgust"],
    key="segmented_control_after_sleep",
)
st.write("segmented-control-after-sleep:", str(s6))

st.segmented_control(
    "Segmented control with content width",
    ["Option 1", "Option 2", "Option 3 with more content"],
    width="content",
    key="segmented_control_content_width",
)

st.segmented_control(
    "Segmented control with stretch width",
    ["Option 1", "Option 2", "Option 3 more content"],
    width="stretch",
    key="segmented_control_stretch_width",
)

st.segmented_control(
    "Segmented control with 300px width",
    ["Option 1", "Option 2", "Option 3 with more content"],
    width=300,
    key="segmented_control_300px_width",
)

if "runs" not in st.session_state:
    st.session_state.runs = 0
st.session_state.runs += 1
st.write("Runs:", st.session_state.runs)

if st.checkbox("Set default values", value=False):
    st.session_state.default_segmented_control_options = [
        "🧰 General widgets",
        "🎥 Video",
    ]
else:
    st.session_state.default_segmented_control_options = []

# The test will only work if this doesn't use a user-specified key:
val = st.segmented_control(
    "Segmented control with default options",
    [
        "🧰 General widgets",
        "📊 Charts",
        "🌇 Images",
        "🎥 Video",
        "📝 Text",
    ],
    selection_mode="multi",
    default=st.session_state.default_segmented_control_options,
)
st.write("Segmented control with default options:", str(val))

st.markdown("Dynamic segmented control:")

if st.toggle("Update segmented control props"):
    dyn_val = st.segmented_control(
        "Updated dynamic segmented control",
        key="dynamic_segmented_control_with_key",
        help="updated help",
        width=300,
        default="banana",
        on_change=lambda a, param: print(
            f"Updated segmented control - callback triggered: {a} {param}"
        ),
        args=("Updated segmented control arg",),
        kwargs={"param": "updated kwarg param"},
        # Whitelisted args:
        options=["apple", "banana", "orange"],
        selection_mode="single",
        format_func=lambda text: text.capitalize(),
    )
    st.write("Updated segmented control value:", dyn_val)
else:
    dyn_val = st.segmented_control(
        "Initial dynamic segmented control",
        key="dynamic_segmented_control_with_key",
        help="initial help",
        width="content",
        default="apple",
        on_change=lambda a, param: print(
            f"Initial segmented control - callback triggered: {a} {param}"
        ),
        args=("Initial segmented control arg",),
        kwargs={"param": "initial kwarg param"},
        # Whitelisted args:
        options=["apple", "banana", "orange"],
        selection_mode="single",
        format_func=lambda text: text.capitalize(),
    )
    st.write("Initial segmented control value:", dyn_val)
