# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import numpy as np
import pandas as pd

import streamlit as st
from streamlit import runtime


def format_option(option: str) -> str:
    return f"Color: {option}"


w1 = st.select_slider(
    "Label 1 (format_func with key and help)",
    value=("orange", "blue"),
    options=["red", "orange", "yellow", "green", "blue", "indigo", "violet"],
    format_func=format_option,
    key="first_select_slider",
    help="Help in a select slider",
)
if "first_select_slider" in st.session_state:
    st.write("Value 1:", st.session_state.first_select_slider)
st.write("Value 1:", w1)

w2 = st.select_slider(
    "Label 2 (no default)",
    options=[1, 2, 3, 4, 5],
)
st.write("Value 2:", w2)

w3 = st.select_slider(
    "Label 3 (default with ints and series)",
    value=[2, 5],
    options=pd.Series([1, 2, 3, 4, 5, 6, 7, 8, 9]),
)
st.write("Value 3:", w3)

w4 = st.select_slider(
    "Label 4 (default with pandas df)",
    value=5,
    options=pd.DataFrame(
        {
            "first column": [1, 2, 3, 4, 5],
            "second column": [10, 20, 30, 40, 50],
        }
    ),
)
st.write("Value 4:", w4)

w5 = st.select_slider(
    "Label 5 (disabled)",
    value=("orange", "blue"),
    options=["red", "orange", "yellow", "green", "blue", "indigo", "violet"],
    disabled=True,
)
st.write("Value 5:", w5)

w6 = st.select_slider(
    "Label 6 (hidden visibility)",
    options=["red", "orange", "yellow", "green", "blue", "indigo", "violet"],
    label_visibility="hidden",
    key="select_slider_hidden",
)

st.write("Value 6:", w6)


w7 = st.select_slider(
    "Label 7 (collapsed visibility)",
    options=["red", "orange", "yellow", "green", "blue", "indigo", "violet"],
    label_visibility="collapsed",
    key="select_slider_collapsed",
)

st.write("Value 7:", w7)

if runtime.exists():

    def on_change():
        st.session_state.select_slider_changed = True
        st.write("Hello world")

    st.select_slider(
        "Label 8 (on change)",
        options=np.array([1, 2, 3, 4, 5]),
        key="select_slider8",
        on_change=on_change,
    )
    st.write("Value 8:", st.session_state.select_slider8)
    st.write("Select slider changed:", "select_slider_changed" in st.session_state)

with st.expander("Expander", expanded=True):
    w9 = st.select_slider(
        label="Label 9 (expander)",
        options=["foo", "bar", "baz", "This is a very, very long option"],
        value="This is a very, very long option",
    )

    st.write("Value 9:", w9)

with st.form(key="my_form", clear_on_submit=True):
    w10 = st.select_slider(
        label="Label 10 (form)",
        options=[1, 2, 3, 4, 5],
    )
    st.form_submit_button("Submit")

st.write("select_slider-in-form selection:", str(w10))


@st.fragment
def test_fragment():
    w11 = st.select_slider(
        label="Label 11 (fragment)",
        options=[1, 2, 3, 4, 5],
    )
    st.write("select_slider-in-fragment selection:", str(w11))


test_fragment()

st.select_slider(
    "Label 12 -> :material/check: :rainbow[Fancy] _**markdown** `label` _support_",
    options=np.array([1, 2, 3, 4, 5]),
)

st.select_slider(
    "Label 13 - Width 300px",
    options=["red", "orange", "yellow", "green", "blue"],
    width=300,
)

st.select_slider(
    "Label 14 - Width Stretch",
    options=["red", "orange", "yellow", "green", "blue"],
    width="stretch",
)

if "runs" not in st.session_state:
    st.session_state.runs = 0
st.session_state.runs += 1
st.write("Runs:", st.session_state.runs)

# Markdown text trick to fix firefox sub-pixel flakiness:
st.write("Dynamic select slider state:")

if st.toggle("Update select slider props"):
    dyn_val = st.select_slider(
        "Updated dynamic select slider",
        value="blue",
        width=300,
        help="updated help",
        key="dynamic_select_slider_with_key",
        on_change=lambda a, param: print(
            f"Updated select slider - callback triggered: {a} {param}"
        ),
        args=("Updated select arg",),
        kwargs={"param": "updated kwarg param"},
        # options are not yet supported for dynamic changes
        # keeping it at the same value:
        options=["red", "orange", "yellow", "green", "blue"],
    )
    st.write("Updated select slider value:", dyn_val)
else:
    dyn_val = st.select_slider(
        "Initial dynamic select slider",
        value="orange",
        width="stretch",
        help="initial help",
        key="dynamic_select_slider_with_key",
        on_change=lambda a, param: print(
            f"Initial select slider - callback triggered: {a} {param}"
        ),
        args=("Initial select arg",),
        kwargs={"param": "initial kwarg param"},
        # options are not yet supported for dynamic changes
        # keeping it at the same value:
        options=["red", "orange", "yellow", "green", "blue"],
    )
    st.write("Initial select slider value:", dyn_val)


MARKDOWN_SELECT_SLIDER_OPTIONS = [
    "~~Strikethrough~~",
    "*Italics*",
    "**Bold**",
    "`Inline code`",
    ":material/check: Icon option",
    ":orange-background[Highlighted text]",
    ":green[Success text]",
    "[Docs](https://streamlit.io)",
]

markdown_select_slider_value = st.select_slider(
    "Label 15 - Markdown in options",
    options=MARKDOWN_SELECT_SLIDER_OPTIONS,
    value=MARKDOWN_SELECT_SLIDER_OPTIONS[0],
    key="markdown_options_select_slider",
)
st.write("Markdown option selection:", markdown_select_slider_value)
