/**
 * Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { screen, within } from "@testing-library/react"

import { Block as BlockProto } from "@streamlit/protobuf"

import { BlockNode } from "~lib/AppNode"
import { render } from "~lib/test_util"

import Tabs, { TabProps } from "./Tabs"

const FAKE_SCRIPT_HASH = "fake_script_hash"

function makeTab(label: string, children: BlockNode[] = []): BlockNode {
  return new BlockNode(
    FAKE_SCRIPT_HASH,
    children,
    new BlockProto({ allowEmpty: true, tab: { label } })
  )
}

function makeTabsNode(tabs: number): BlockNode {
  return new BlockNode(
    FAKE_SCRIPT_HASH,
    Array.from({ length: tabs }, (_element, index) => makeTab(`Tab ${index}`)),
    new BlockProto({ allowEmpty: true })
  )
}

const getProps = (props?: Partial<TabProps>): TabProps =>
  Object({
    widgetsDisabled: false,
    node: makeTabsNode(5),
    isStale: false,
    ...props,
    renderTabContent: vi.fn(),
  })

describe("st.tabs", () => {
  it("renders without crashing", () => {
    render(<Tabs {...getProps()} />)

    const tabsElement = screen.getByTestId("stTabs")
    expect(tabsElement).toBeInTheDocument()
    expect(tabsElement).toHaveClass("stTabs")

    const tabsContainer = screen.getByRole("tablist")
    expect(tabsContainer).toBeInTheDocument()
    const tabs = within(tabsContainer).getAllByRole("tab")
    expect(tabs).toHaveLength(5)
  })

  it("sets the tab labels correctly", () => {
    render(<Tabs {...getProps()} />)
    const tabs = screen.getAllByRole("tab")
    expect(tabs).toHaveLength(5)

    tabs.forEach((tab, index) => {
      expect(tab).toHaveTextContent(`Tab ${index}`)
    })
  })

  it("sets the correct default tab index", () => {
    const node = makeTabsNode(3)
    node.deltaBlock.tabContainer = { defaultTabIndex: 2 }

    render(<Tabs {...getProps({ node })} />)

    const tabs = screen.getAllByRole("tab")
    expect(tabs[2]).toHaveAttribute("aria-selected", "true")
  })

  it("selects the first occurrence when default points to a duplicate label", () => {
    const node = new BlockNode(
      FAKE_SCRIPT_HASH,
      [makeTab("Unique"), makeTab("Dupe"), makeTab("Dupe")],
      new BlockProto({ allowEmpty: true })
    )
    node.deltaBlock.tabContainer = { defaultTabIndex: 1 }

    render(<Tabs {...getProps({ node })} />)

    const tabs = screen.getAllByRole("tab")
    expect(tabs[1]).toHaveAttribute("aria-selected", "true")
  })

  it("doesn't disable tabs when widgets are disabled", () => {
    render(<Tabs {...getProps({ widgetsDisabled: true })} />)
    const tabs = screen.getAllByRole("tab")

    tabs.forEach((_, index) => {
      // the selected tab does not have the disabled prop as true in baseweb
      if (index == 0) {
        return
      }
      expect(tabs[index]).not.toBeDisabled()
    })
  })
})
