/**
 * Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import {
  GridCell,
  GridCellKind,
  LoadingCell,
} from "@glideapps/glide-data-grid"
import { RangeCellType } from "@glideapps/glide-data-grid-cells"

import { isIntegerType } from "~lib/dataframes/arrowTypeUtils"
import { EmotionTheme } from "~lib/theme"
import { formatNumber } from "~lib/util/formatNumber"
import { isNullOrUndefined, notNullOrUndefined } from "~lib/util/utils"

import {
  BaseColumn,
  BaseColumnProps,
  countDecimals,
  getEmptyCell,
  getErrorCell,
  mergeColumnParameters,
  toSafeNumber,
  toSafeString,
} from "./utils"

type ChartAutoColor = "auto" | "auto-inverse"
type ChartNamedSwatch =
  | "red"
  | "blue"
  | "green"
  | "yellow"
  | "violet"
  | "orange"
  | "gray"
  | "grey"
  | "primary"

declare const __cssColorBrand: unique symbol
type CSSColorString = string & { readonly [__cssColorBrand]?: never }

type ChartColor = ChartAutoColor | ChartNamedSwatch | CSSColorString

/**
 * Get the color mapping to map a user-defined color to the our
 * theme colors.
 *
 * @param theme The theme to use.
 * @returns The color mapping.
 */
const getColorMapping = (theme: EmotionTheme): Map<string, string> => {
  return new Map(
    Object.entries({
      red: theme.colors.redColor,
      blue: theme.colors.blueColor,
      green: theme.colors.greenColor,
      yellow: theme.colors.yellowColor,
      violet: theme.colors.violetColor,
      orange: theme.colors.orangeColor,
      gray: theme.colors.grayColor,
      grey: theme.colors.grayColor,
      primary: theme.colors.primary,
    })
  )
}
export interface ProgressColumnParams {
  /**
   * The minimum permitted value. Defaults to 0.
   */
  readonly min_value?: number
  /**
   * The maximum permitted value. Defaults to 100 if the underlying data is
   * integer, or 1 for all others types.
   */
  readonly max_value?: number
  /**
   * A formatting syntax (e.g. sprintf) to format the display value.
   * This can be used for adding prefix or suffix, or changing the number of
   * decimals of the display value.
   */
  readonly format?: string
  /**
   * The stepping interval. Defaults to 0.01.
   * Mainly useful once we provide editing capabilities.
   */
  readonly step?: number
  /**
   * The color to use for the progress bar. Can be:
   * - auto & auto-inverse: To color the bar green or red depending on the value.
   * - red, blue, green, yellow, orange, violet, gray/grey, primary (from theme)
   * - a CSS color value compatible with canvas rendering
   */
  readonly color?: ChartColor
}

/**
 * A read-only column type to support rendering values that have a defined
 * range. This is rendered via a progress-bar-like visualization.
 */
function ProgressColumn(
  props: BaseColumnProps,
  theme: EmotionTheme
): BaseColumn {
  const isInteger = isIntegerType(props.arrowType)

  const parameters = mergeColumnParameters(
    // Default parameters:
    {
      min_value: 0,
      max_value: isInteger ? 100 : 1,
      format: isInteger ? "%3d%%" : "percent",
      step: isInteger ? 1 : undefined,
      color: undefined,
    } as ProgressColumnParams,
    // User parameters:
    props.columnTypeOptions
  ) as ProgressColumnParams

  const colorMapping = getColorMapping(theme)

  const fixedDecimals =
    isNullOrUndefined(parameters.step) || Number.isNaN(parameters.step)
      ? undefined
      : countDecimals(parameters.step)

  // Measure the display value of the max value, so that all progress bars are aligned correctly:
  let measureLabel: string
  try {
    measureLabel = formatNumber(
      parameters.max_value as number,
      parameters.format,
      fixedDecimals
    )
  } catch {
    measureLabel = toSafeString(parameters.max_value)
  }

  const cellTemplate: RangeCellType = {
    kind: GridCellKind.Custom,
    allowOverlay: false,
    copyData: "",
    contentAlign: props.contentAlignment,
    readonly: true,
    data: {
      kind: "range-cell",
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      min: parameters.min_value!,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      max: parameters.max_value!,
      step: parameters.step ?? 0.01,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      value: parameters.min_value!,
      label: String(parameters.min_value),
      measureLabel,
    },
  }

  return {
    ...props,
    kind: "progress",
    sortMode: "smart",
    typeIcon: ":material/commit:",
    isEditable: false, // Progress column is always readonly
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- TODO: Replace 'any' with a more specific type.
    getCell(data?: any): GridCell {
      if (isNullOrUndefined(data)) {
        // TODO(lukasmasuch): Use a missing cell?
        return getEmptyCell()
      }

      if (
        isNullOrUndefined(parameters.min_value) ||
        isNullOrUndefined(parameters.max_value) ||
        Number.isNaN(parameters.min_value) ||
        Number.isNaN(parameters.max_value) ||
        parameters.min_value >= parameters.max_value
      ) {
        return getErrorCell(
          "Invalid min/max parameters",
          `The min_value (${parameters.min_value}) and max_value (${parameters.max_value}) parameters must be valid numbers.`
        )
      }

      if (
        notNullOrUndefined(parameters.step) &&
        Number.isNaN(parameters.step)
      ) {
        return getErrorCell(
          "Invalid step parameter",
          `The step parameter (${parameters.step}) must be a valid number.`
        )
      }

      const cellData = toSafeNumber(data)

      if (Number.isNaN(cellData) || isNullOrUndefined(cellData)) {
        return getErrorCell(
          toSafeString(data),
          "The value cannot be interpreted as a number."
        )
      }

      // Check if the value is larger than the maximum supported value:
      if (Number.isInteger(cellData) && !Number.isSafeInteger(cellData)) {
        return getErrorCell(
          toSafeString(data),
          "The value is larger than the maximum supported integer values in number columns (2^53)."
        )
      }

      let displayData = ""

      try {
        displayData = formatNumber(cellData, parameters.format, fixedDecimals)
      } catch (error) {
        return getErrorCell(
          toSafeString(cellData),
          notNullOrUndefined(parameters.format)
            ? // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
              `Failed to format the number based on the provided format configuration: (${parameters.format}). Error: ${error}`
            : // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
              `Failed to format the number. Error: ${error}`
        )
      }

      // If the value is outside the range, we scale it to the min/max
      // for the visualization.
      const normalizeCellValue = Math.min(
        parameters.max_value,
        Math.max(parameters.min_value, cellData)
      )

      // Determine progress color if configured
      let progressColor: string | undefined
      if (parameters.color === "auto" || parameters.color === "auto-inverse") {
        // Use min/max to determine threshold at 50%
        const range = parameters.max_value - parameters.min_value
        const ratio =
          range === 0 ? 0 : (normalizeCellValue - parameters.min_value) / range
        const isAbove = ratio > 0.5
        if (parameters.color === "auto") {
          progressColor = isAbove
            ? theme?.colors.greenColor
            : theme?.colors.redColor
        } else {
          progressColor = isAbove
            ? theme?.colors.redColor
            : theme?.colors.greenColor
        }
      } else if (parameters.color) {
        progressColor =
          colorMapping.get(parameters.color) ?? (parameters.color as string)
      }

      return {
        ...cellTemplate,
        isMissingValue: isNullOrUndefined(data),
        copyData: String(cellData), // Column sorting is done via the copyData value
        data: {
          ...cellTemplate.data,
          value: normalizeCellValue,
          label: displayData,
          measureLabel:
            displayData.length > measureLabel.length
              ? // Use displayData if it's longer than measureLabel to determine
                // the width of the progress bar label.
                displayData
              : measureLabel,
          color: progressColor,
        },
      } as RangeCellType
    },
    getCellValue(cell: RangeCellType | LoadingCell): number | null {
      if (cell.kind === GridCellKind.Loading) {
        return null
      }
      return cell.data?.value === undefined ? null : cell.data?.value
    },
  }
}

ProgressColumn.isEditableType = false

export default ProgressColumn
