/**
 * Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import {
  BYTE_CONVERSION_SIZE,
  FileSize,
  getSizeDisplay,
  sizeConverter,
} from "./FileHelper"

describe("getSizeDisplay", () => {
  it("shows unit", () => {
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE, FileSize.Byte)).toEqual(
      "1.0KB"
    )
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE ** 2, FileSize.Byte)).toEqual(
      "1.0MB"
    )
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE ** 3, FileSize.Byte)).toEqual(
      "1.0GB"
    )

    expect(getSizeDisplay(10, FileSize.Gigabyte)).toEqual("10.0GB")
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE, FileSize.Megabyte)).toEqual(
      "1.0GB"
    )
  })

  it("has unusual values", () => {
    expect(() => getSizeDisplay(-100, FileSize.Byte)).toThrow(
      "Size must be greater than or equal to 0"
    )
    expect(getSizeDisplay(0, FileSize.Byte, -1)).toEqual("0B")
  })

  it("truncates to the right amount of decimals", () => {
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE, FileSize.Byte)).toEqual(
      "1.0KB"
    )
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE, FileSize.Byte, 0)).toEqual(
      "1KB"
    )
    expect(getSizeDisplay(BYTE_CONVERSION_SIZE, FileSize.Byte, 3)).toEqual(
      "1.000KB"
    )
  })

  it("rounds up to the next unit", () => {
    expect(getSizeDisplay(500, FileSize.Byte)).toEqual("500.0B")
    expect(getSizeDisplay(800, FileSize.Byte)).toEqual("0.8KB")
    expect(getSizeDisplay(501, FileSize.Gigabyte)).toEqual("501.0GB")
  })
})

describe("sizeConverter", () => {
  it("converts up to the bigger unit", () => {
    expect(sizeConverter(0.5, FileSize.Kilobyte, FileSize.Megabyte)).toEqual(
      0.5 / BYTE_CONVERSION_SIZE
    )
    expect(
      sizeConverter(BYTE_CONVERSION_SIZE, FileSize.Byte, FileSize.Kilobyte)
    ).toEqual(1)
    expect(
      sizeConverter(
        BYTE_CONVERSION_SIZE ** 2,
        FileSize.Kilobyte,
        FileSize.Gigabyte
      )
    ).toEqual(1)
    expect(sizeConverter(1, FileSize.Megabyte, FileSize.Gigabyte)).toEqual(
      1 / BYTE_CONVERSION_SIZE
    )
  })

  it("converts down to the smaller unit", () => {
    expect(sizeConverter(0.5, FileSize.Gigabyte, FileSize.Megabyte)).toEqual(
      BYTE_CONVERSION_SIZE * 0.5
    )
    expect(
      sizeConverter(BYTE_CONVERSION_SIZE, FileSize.Gigabyte, FileSize.Kilobyte)
    ).toEqual(BYTE_CONVERSION_SIZE ** 3)
    expect(
      sizeConverter(
        BYTE_CONVERSION_SIZE ** 2,
        FileSize.Megabyte,
        FileSize.Kilobyte
      )
    ).toEqual(BYTE_CONVERSION_SIZE ** 3)
    expect(sizeConverter(1, FileSize.Kilobyte, FileSize.Byte)).toEqual(
      BYTE_CONVERSION_SIZE
    )
  })

  it("handles unusual cases", () => {
    expect(
      sizeConverter(BYTE_CONVERSION_SIZE, FileSize.Byte, FileSize.Byte)
    ).toEqual(BYTE_CONVERSION_SIZE)
    expect(() =>
      sizeConverter(-1, FileSize.Gigabyte, FileSize.Gigabyte)
    ).toThrow("Size must be 0 or greater")
  })
})
