//! Stores a LIVE SELECT query definition on the table
use std::borrow::Cow;

use anyhow::Result;
use storekey::{BorrowDecode, Encode};
use uuid::Uuid;

use crate::catalog::{DatabaseId, NamespaceId, SubscriptionDefinition};
use crate::key::category::{Categorise, Category};
use crate::kvs::{KVKey, impl_kv_key_storekey};
use crate::val::TableName;

/// Lv is used to track a live query and is cluster independent, i.e. it is tied
/// with a ns/db/tb combo without the cl. The live statement includes the node
/// id, so lq can be derived purely from an lv.
///
/// The value of the lv is the statement.
#[derive(Clone, Debug, Eq, PartialEq, PartialOrd, Encode, BorrowDecode)]
#[storekey(format = "()")]
pub(crate) struct Lq<'a> {
	__: u8,
	_a: u8,
	pub ns: NamespaceId,
	_b: u8,
	pub db: DatabaseId,
	_c: u8,
	pub tb: Cow<'a, TableName>,
	_d: u8,
	_e: u8,
	_f: u8,
	pub lq: Uuid,
}

impl_kv_key_storekey!(Lq<'_> => SubscriptionDefinition);

pub fn new(ns: NamespaceId, db: DatabaseId, tb: &TableName, lq: Uuid) -> Lq<'_> {
	Lq::new(ns, db, tb, lq)
}

pub fn prefix(ns: NamespaceId, db: DatabaseId, tb: &TableName) -> Result<Vec<u8>> {
	let mut k = super::all::new(ns, db, tb).encode_key()?;
	k.extend_from_slice(b"!lq\x00");
	Ok(k)
}

pub fn suffix(ns: NamespaceId, db: DatabaseId, tb: &TableName) -> Result<Vec<u8>> {
	let mut k = super::all::new(ns, db, tb).encode_key()?;
	k.extend_from_slice(b"!lq\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00");
	Ok(k)
}

impl Categorise for Lq<'_> {
	fn categorise(&self) -> Category {
		Category::TableLiveQuery
	}
}

impl<'a> Lq<'a> {
	pub fn new(ns: NamespaceId, db: DatabaseId, tb: &'a TableName, lq: Uuid) -> Self {
		Self {
			__: b'/',
			_a: b'*',
			ns,
			_b: b'*',
			db,
			_c: b'*',
			tb: Cow::Borrowed(tb),
			_d: b'!',
			_e: b'l',
			_f: b'q',
			lq,
		}
	}

	pub fn decode_key(k: &[u8]) -> anyhow::Result<Lq<'_>> {
		Ok(storekey::decode_borrow(k)?)
	}
}

#[cfg(test)]
mod tests {
	use super::*;

	#[test]
	fn key() {
		let live_query_id =
			Uuid::from_bytes([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
		let tb = TableName::from("testtb");
		let val = Lq::new(NamespaceId(1), DatabaseId(2), &tb, live_query_id);
		let enc = Lq::encode_key(&val).unwrap();
		assert_eq!(
			enc,
			b"/*\x00\x00\x00\x01*\x00\x00\x00\x02*testtb\0!lq\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
		);
	}

	#[test]
	fn prefix() {
		let tb = TableName::from("testtb");
		let val = super::prefix(NamespaceId(1), DatabaseId(2), &tb).unwrap();
		assert_eq!(val, b"/*\x00\x00\x00\x01*\x00\x00\x00\x02*testtb\0!lq\x00")
	}

	#[test]
	fn suffix() {
		let tb = TableName::from("testtb");
		let val = super::suffix(NamespaceId(1), DatabaseId(2), &tb).unwrap();
		assert_eq!(val, b"/*\x00\x00\x00\x01*\x00\x00\x00\x02*testtb\0!lq\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00")
	}
}
