use phf::{phf_map, phf_set};
use unicase::UniCase;

use crate::sql::Algorithm;
use crate::sql::language::Language;
use crate::syn::token::{DistanceKind, Keyword, TokenKind, VectorTypeKind};

/// A set of keywords which might in some contexts are dissallowed as an
/// identifier.
pub static RESERVED_KEYWORD: phf::Set<UniCase<&'static str>> = phf_set! {
	UniCase::ascii("ALTER"),
	UniCase::ascii("BEGIN"),
	UniCase::ascii("BREAK"),
	UniCase::ascii("CANCEL"),
	UniCase::ascii("COMMIT"),
	UniCase::ascii("CONTINUE"),
	UniCase::ascii("CREATE"),
	UniCase::ascii("DEFINE"),
	UniCase::ascii("FOR"),
	UniCase::ascii("IF"),
	UniCase::ascii("INFO"),
	UniCase::ascii("INSERT"),
	UniCase::ascii("KILL"),
	UniCase::ascii("LIVE"),
	UniCase::ascii("OPTION"),
	UniCase::ascii("REBUILD"),
	UniCase::ascii("RETURN"),
	UniCase::ascii("RELATE"),
	UniCase::ascii("REMOVE"),
	UniCase::ascii("SELECT"),
	UniCase::ascii("LET"),
	UniCase::ascii("SHOW"),
	UniCase::ascii("SLEEP"),
	UniCase::ascii("THROW"),
	UniCase::ascii("UPDATE"),
	UniCase::ascii("UPSERT"),
	UniCase::ascii("USE"),
	UniCase::ascii("DIFF"),
	UniCase::ascii("RAND"),
	UniCase::ascii("NONE"),
	UniCase::ascii("NULL"),
	UniCase::ascii("AFTER"),
	UniCase::ascii("BEFORE"),
	UniCase::ascii("VALUE"),
	UniCase::ascii("BY"),
	UniCase::ascii("ALL"),
	UniCase::ascii("TRUE"),
	UniCase::ascii("FALSE"),
	UniCase::ascii("WHERE"),
	UniCase::ascii("TABLE"),
	UniCase::ascii("SEQUENCE"),
	UniCase::ascii("FUNCTION"),
};

pub fn could_be_reserved(s: &str) -> bool {
	RESERVED_KEYWORD.contains(&UniCase::ascii(s))
}

/// A map for mapping keyword strings to a tokenkind,
pub(crate) static KEYWORDS: phf::Map<UniCase<&'static str>, TokenKind> = phf_map! {
	// Keywords
	UniCase::ascii("ACCESS") => TokenKind::Keyword(Keyword::Access),
	UniCase::ascii("AFTER") => TokenKind::Keyword(Keyword::After),
	UniCase::ascii("ALGORITHM") => TokenKind::Keyword(Keyword::Algorithm),
	UniCase::ascii("ALL") => TokenKind::Keyword(Keyword::All),
	UniCase::ascii("ALLINSIDE") => TokenKind::Keyword(Keyword::AllInside),
	UniCase::ascii("ALTER") => TokenKind::Keyword(Keyword::Alter),
	UniCase::ascii("ALWAYS") => TokenKind::Keyword(Keyword::Always),
	UniCase::ascii("ANALYZER") => TokenKind::Keyword(Keyword::Analyzer),
	UniCase::ascii("AND") => TokenKind::Keyword(Keyword::And),
	UniCase::ascii("ANDKW") => TokenKind::Keyword(Keyword::AndKw),
	UniCase::ascii("ANYINSIDE") => TokenKind::Keyword(Keyword::AnyInside),
	UniCase::ascii("API") => TokenKind::Keyword(Keyword::Api),
	UniCase::ascii("AS") => TokenKind::Keyword(Keyword::As),
	UniCase::ascii("ASC") => TokenKind::Keyword(Keyword::Ascending),
	UniCase::ascii("ASCENDING") => TokenKind::Keyword(Keyword::Ascending),
	UniCase::ascii("ASCII") => TokenKind::Keyword(Keyword::Ascii),
	UniCase::ascii("ASSERT") => TokenKind::Keyword(Keyword::Assert),
	UniCase::ascii("AT") => TokenKind::Keyword(Keyword::At),
	UniCase::ascii("AUTHENTICATE") => TokenKind::Keyword(Keyword::Authenticate),
	UniCase::ascii("AUTO") => TokenKind::Keyword(Keyword::Auto),
	UniCase::ascii("BACKEND") => TokenKind::Keyword(Keyword::Backend),
	UniCase::ascii("BATCH") => TokenKind::Keyword(Keyword::Batch),
	UniCase::ascii("BEARER") => TokenKind::Keyword(Keyword::Bearer),
	UniCase::ascii("BEFORE") => TokenKind::Keyword(Keyword::Before),
	UniCase::ascii("BEGIN") => TokenKind::Keyword(Keyword::Begin),
	UniCase::ascii("BLANK") => TokenKind::Keyword(Keyword::Blank),
	UniCase::ascii("BM25") => TokenKind::Keyword(Keyword::Bm25),
	UniCase::ascii("BREAK") => TokenKind::Keyword(Keyword::Break),
	UniCase::ascii("BUCKET") => TokenKind::Keyword(Keyword::Bucket),
	UniCase::ascii("BY") => TokenKind::Keyword(Keyword::By),
	UniCase::ascii("CAMEL") => TokenKind::Keyword(Keyword::Camel),
	UniCase::ascii("CANCEL") => TokenKind::Keyword(Keyword::Cancel),
	UniCase::ascii("CAPACITY") => TokenKind::Keyword(Keyword::Capacity),
	UniCase::ascii("CASCADE") => TokenKind::Keyword(Keyword::Cascade),
	UniCase::ascii("CHANGEFEED") => TokenKind::Keyword(Keyword::ChangeFeed),
	UniCase::ascii("CHANGES") => TokenKind::Keyword(Keyword::Changes),
	UniCase::ascii("CLASS") => TokenKind::Keyword(Keyword::Class),
	UniCase::ascii("COLLATE") => TokenKind::Keyword(Keyword::Collate),
	UniCase::ascii("COLUMNS") => TokenKind::Keyword(Keyword::Fields),
	UniCase::ascii("COMMENT") => TokenKind::Keyword(Keyword::Comment),
	UniCase::ascii("COMMIT") => TokenKind::Keyword(Keyword::Commit),
	UniCase::ascii("COMPACT") => TokenKind::Keyword(Keyword::Compact),
	UniCase::ascii("COMPUTED") => TokenKind::Keyword(Keyword::Computed),
	UniCase::ascii("CONCURRENTLY") => TokenKind::Keyword(Keyword::Concurrently),
	UniCase::ascii("CONFIG") => TokenKind::Keyword(Keyword::Config),
	UniCase::ascii("CONTAINS") => TokenKind::Keyword(Keyword::Contains),
	UniCase::ascii("CONTAINSALL") => TokenKind::Keyword(Keyword::ContainsAll),
	UniCase::ascii("CONTAINSANY") => TokenKind::Keyword(Keyword::ContainsAny),
	UniCase::ascii("CONTAINSNONE") => TokenKind::Keyword(Keyword::ContainsNone),
	UniCase::ascii("CONTAINSNOT") => TokenKind::Keyword(Keyword::ContainsNot),
	UniCase::ascii("CONTENT") => TokenKind::Keyword(Keyword::Content),
	UniCase::ascii("CONTINUE") => TokenKind::Keyword(Keyword::Continue),
	UniCase::ascii("COUNT") => TokenKind::Keyword(Keyword::Count),
	UniCase::ascii("CREATE") => TokenKind::Keyword(Keyword::Create),
	UniCase::ascii("DATABASE") => TokenKind::Keyword(Keyword::Database),
	UniCase::ascii("DB") => TokenKind::Keyword(Keyword::Database),
	UniCase::ascii("DEFAULT") => TokenKind::Keyword(Keyword::Default),
	UniCase::ascii("DEFINE") => TokenKind::Keyword(Keyword::Define),
	UniCase::ascii("DELETE") => TokenKind::Keyword(Keyword::Delete),
	UniCase::ascii("DESC") => TokenKind::Keyword(Keyword::Descending),
	UniCase::ascii("DESCENDING") => TokenKind::Keyword(Keyword::Descending),
	UniCase::ascii("DIFF") => TokenKind::Keyword(Keyword::Diff),
	UniCase::ascii("DIMENSION") => TokenKind::Keyword(Keyword::Dimension),
	UniCase::ascii("DIST") => TokenKind::Keyword(Keyword::Distance),
	UniCase::ascii("DISTANCE") => TokenKind::Keyword(Keyword::Distance),
	UniCase::ascii("DROP") => TokenKind::Keyword(Keyword::Drop),
	UniCase::ascii("DUPLICATE") => TokenKind::Keyword(Keyword::Duplicate),
	UniCase::ascii("EDGENGRAM") => TokenKind::Keyword(Keyword::Edgengram),
	UniCase::ascii("EFC") => TokenKind::Keyword(Keyword::Efc),
	UniCase::ascii("ELSE") => TokenKind::Keyword(Keyword::Else),
	UniCase::ascii("END") => TokenKind::Keyword(Keyword::End),
	UniCase::ascii("ENFORCED") => TokenKind::Keyword(Keyword::Enforced),
	UniCase::ascii("EVENT") => TokenKind::Keyword(Keyword::Event),
	UniCase::ascii("EXCLUDE") => TokenKind::Keyword(Keyword::Exclude),
	UniCase::ascii("EXISTS") => TokenKind::Keyword(Keyword::Exists),
	UniCase::ascii("EXPIRED") => TokenKind::Keyword(Keyword::Expired),
	UniCase::ascii("EXPLAIN") => TokenKind::Keyword(Keyword::Explain),
	UniCase::ascii("EXPUNGE") => TokenKind::Keyword(Keyword::Expunge),
	UniCase::ascii("EXTEND_CANDIDATES") => TokenKind::Keyword(Keyword::ExtendCandidates),
	UniCase::ascii("false") => TokenKind::Keyword(Keyword::False),
	UniCase::ascii("FETCH") => TokenKind::Keyword(Keyword::Fetch),
	UniCase::ascii("FIELD") => TokenKind::Keyword(Keyword::Field),
	UniCase::ascii("FIELDS") => TokenKind::Keyword(Keyword::Fields),
	UniCase::ascii("FILTERS") => TokenKind::Keyword(Keyword::Filters),
	UniCase::ascii("FLEX") => TokenKind::Keyword(Keyword::Flexible),
	UniCase::ascii("FLEXI") => TokenKind::Keyword(Keyword::Flexible),
	UniCase::ascii("FLEXIBLE") => TokenKind::Keyword(Keyword::Flexible),
	UniCase::ascii("FOR") => TokenKind::Keyword(Keyword::For),
	UniCase::ascii("FROM") => TokenKind::Keyword(Keyword::From),
	UniCase::ascii("FULL") => TokenKind::Keyword(Keyword::Full),
	UniCase::ascii("FULLTEXT") => TokenKind::Keyword(Keyword::Fulltext),
	UniCase::ascii("FUNCTION") => TokenKind::Keyword(Keyword::Function),
	UniCase::ascii("FUNCTIONS") => TokenKind::Keyword(Keyword::Functions),
	UniCase::ascii("GRANT") => TokenKind::Keyword(Keyword::Grant),
	UniCase::ascii("GRAPHQL") => TokenKind::Keyword(Keyword::Graphql),
	UniCase::ascii("GROUP") => TokenKind::Keyword(Keyword::Group),
	UniCase::ascii("HEADERS") => TokenKind::Keyword(Keyword::Headers),
	UniCase::ascii("HIGHLIGHTS") => TokenKind::Keyword(Keyword::Highlights),
	UniCase::ascii("HNSW") => TokenKind::Keyword(Keyword::Hnsw),
	UniCase::ascii("IF") => TokenKind::Keyword(Keyword::If),
	UniCase::ascii("IGNORE") => TokenKind::Keyword(Keyword::Ignore),
	UniCase::ascii("IN") => TokenKind::Keyword(Keyword::In),
	UniCase::ascii("INCLUDE") => TokenKind::Keyword(Keyword::Include),
	UniCase::ascii("INDEX") => TokenKind::Keyword(Keyword::Index),
	UniCase::ascii("INFO") => TokenKind::Keyword(Keyword::Info),
	UniCase::ascii("INSERT") => TokenKind::Keyword(Keyword::Insert),
	UniCase::ascii("INSIDE") => TokenKind::Keyword(Keyword::Inside),
	UniCase::ascii("INTERSECTS") => TokenKind::Keyword(Keyword::Intersects),
	UniCase::ascii("INTO") => TokenKind::Keyword(Keyword::Into),
	UniCase::ascii("IS") => TokenKind::Keyword(Keyword::Is),
	UniCase::ascii("ISSUER") => TokenKind::Keyword(Keyword::Issuer),
	UniCase::ascii("JWKS") => TokenKind::Keyword(Keyword::Jwks),
	UniCase::ascii("JWT") => TokenKind::Keyword(Keyword::Jwt),
	UniCase::ascii("KEEP_PRUNED_CONNECTIONS") => TokenKind::Keyword(Keyword::KeepPrunedConnections),
	UniCase::ascii("KEY") => TokenKind::Keyword(Keyword::Key),
	UniCase::ascii("KILL") => TokenKind::Keyword(Keyword::Kill),
	UniCase::ascii("KV") => TokenKind::Keyword(Keyword::Root),
	UniCase::ascii("LET") => TokenKind::Keyword(Keyword::Let),
	UniCase::ascii("LIMIT") => TokenKind::Keyword(Keyword::Limit),
	UniCase::ascii("LIVE") => TokenKind::Keyword(Keyword::Live),
	UniCase::ascii("LM") => TokenKind::Keyword(Keyword::Lm),
	UniCase::ascii("LOWERCASE") => TokenKind::Keyword(Keyword::Lowercase),
	UniCase::ascii("M") => TokenKind::Keyword(Keyword::M),
	UniCase::ascii("M0") => TokenKind::Keyword(Keyword::M0),
	UniCase::ascii("MAPPER") => TokenKind::Keyword(Keyword::Mapper),
	UniCase::ascii("MERGE") => TokenKind::Keyword(Keyword::Merge),
	UniCase::ascii("MIDDLEWARE") => TokenKind::Keyword(Keyword::Middleware),
	UniCase::ascii("ML") => TokenKind::Keyword(Keyword::ML),
	UniCase::ascii("MODEL") => TokenKind::Keyword(Keyword::Model),
	UniCase::ascii("MODULE") => TokenKind::Keyword(Keyword::Module),
	UniCase::ascii("NAMESPACE") => TokenKind::Keyword(Keyword::Namespace),
	UniCase::ascii("NGRAM") => TokenKind::Keyword(Keyword::Ngram),
	UniCase::ascii("NO") => TokenKind::Keyword(Keyword::No),
	UniCase::ascii("NOINDEX") => TokenKind::Keyword(Keyword::NoIndex),
	UniCase::ascii("NONE") => TokenKind::Keyword(Keyword::None),
	UniCase::ascii("NONEINSIDE") => TokenKind::Keyword(Keyword::NoneInside),
	UniCase::ascii("NORMAL") => TokenKind::Keyword(Keyword::Normal),
	UniCase::ascii("NOT") => TokenKind::Keyword(Keyword::Not),
	UniCase::ascii("NOTINSIDE") => TokenKind::Keyword(Keyword::NotInside),
	UniCase::ascii("NS") => TokenKind::Keyword(Keyword::Namespace),
	UniCase::ascii("NULL") => TokenKind::Keyword(Keyword::Null),
	UniCase::ascii("NUMERIC") => TokenKind::Keyword(Keyword::Numeric),
	UniCase::ascii("OMIT") => TokenKind::Keyword(Keyword::Omit),
	UniCase::ascii("ON") => TokenKind::Keyword(Keyword::On),
	UniCase::ascii("ONLY") => TokenKind::Keyword(Keyword::Only),
	UniCase::ascii("OPTION") => TokenKind::Keyword(Keyword::Option),
	UniCase::ascii("OR") => TokenKind::Keyword(Keyword::OrKw),
	UniCase::ascii("ORDER") => TokenKind::Keyword(Keyword::Order),
	UniCase::ascii("ORIGINAL") => TokenKind::Keyword(Keyword::Original),
	UniCase::ascii("OUT") => TokenKind::Keyword(Keyword::Out),
	UniCase::ascii("OUTSIDE") => TokenKind::Keyword(Keyword::Outside),
	UniCase::ascii("OVERWRITE") => TokenKind::Keyword(Keyword::Overwrite),
	UniCase::ascii("PARALLEL") => TokenKind::Keyword(Keyword::Parallel),
	UniCase::ascii("PARAM") => TokenKind::Keyword(Keyword::Param),
	UniCase::ascii("PASSHASH") => TokenKind::Keyword(Keyword::Passhash),
	UniCase::ascii("PASSWORD") => TokenKind::Keyword(Keyword::Password),
	UniCase::ascii("PATCH") => TokenKind::Keyword(Keyword::Patch),
	UniCase::ascii("PERMISSIONS") => TokenKind::Keyword(Keyword::Permissions),
	UniCase::ascii("POSTINGS_CACHE") => TokenKind::Keyword(Keyword::PostingsCache),
	UniCase::ascii("POSTINGS_ORDER") => TokenKind::Keyword(Keyword::PostingsOrder),
	UniCase::ascii("PREPARE") => TokenKind::Keyword(Keyword::Prepare),
	UniCase::ascii("PUNCT") => TokenKind::Keyword(Keyword::Punct),
	UniCase::ascii("PURGE") => TokenKind::Keyword(Keyword::Purge),
	UniCase::ascii("RANGE") => TokenKind::Keyword(Keyword::Range),
	UniCase::ascii("READONLY") => TokenKind::Keyword(Keyword::Readonly),
	UniCase::ascii("REBUILD") => TokenKind::Keyword(Keyword::Rebuild),
	UniCase::ascii("REFERENCE") => TokenKind::Keyword(Keyword::Reference),
	UniCase::ascii("REFRESH") => TokenKind::Keyword(Keyword::Refresh),
	UniCase::ascii("REJECT") => TokenKind::Keyword(Keyword::Reject),
	UniCase::ascii("RELATE") => TokenKind::Keyword(Keyword::Relate),
	UniCase::ascii("RELATION") => TokenKind::Keyword(Keyword::Relation),
	UniCase::ascii("REMOVE") => TokenKind::Keyword(Keyword::Remove),
	UniCase::ascii("REPLACE") => TokenKind::Keyword(Keyword::Replace),
	UniCase::ascii("RETURN") => TokenKind::Keyword(Keyword::Return),
	UniCase::ascii("REVOKE") => TokenKind::Keyword(Keyword::Revoke),
	UniCase::ascii("REVOKED") => TokenKind::Keyword(Keyword::Revoked),
	UniCase::ascii("ROLES") => TokenKind::Keyword(Keyword::Roles),
	UniCase::ascii("ROOT") => TokenKind::Keyword(Keyword::Root),
	UniCase::ascii("SC") => TokenKind::Keyword(Keyword::Scope),
	UniCase::ascii("SCHEMAFUL") => TokenKind::Keyword(Keyword::Schemafull),
	UniCase::ascii("SCHEMAFULL") => TokenKind::Keyword(Keyword::Schemafull),
	UniCase::ascii("SCHEMALESS") => TokenKind::Keyword(Keyword::Schemaless),
	UniCase::ascii("SELECT") => TokenKind::Keyword(Keyword::Select),
	UniCase::ascii("SEQUENCE") => TokenKind::Keyword(Keyword::Sequence),
	UniCase::ascii("SESSION") => TokenKind::Keyword(Keyword::Session),
	UniCase::ascii("SET") => TokenKind::Keyword(Keyword::Set),
	UniCase::ascii("SHOW") => TokenKind::Keyword(Keyword::Show),
	UniCase::ascii("SIGNIN") => TokenKind::Keyword(Keyword::Signin),
	UniCase::ascii("SIGNUP") => TokenKind::Keyword(Keyword::Signup),
	UniCase::ascii("SINCE") => TokenKind::Keyword(Keyword::Since),
	UniCase::ascii("SLEEP") => TokenKind::Keyword(Keyword::Sleep),
	UniCase::ascii("SNOWBALL") => TokenKind::Keyword(Keyword::Snowball),
	UniCase::ascii("SPLIT") => TokenKind::Keyword(Keyword::Split),
	UniCase::ascii("START") => TokenKind::Keyword(Keyword::Start),
	UniCase::ascii("STRICT") => TokenKind::Keyword(Keyword::Strict),
	UniCase::ascii("STRUCTURE") => TokenKind::Keyword(Keyword::Structure),
	UniCase::ascii("SYSTEM") => TokenKind::Keyword(Keyword::System),
	UniCase::ascii("TABLE") => TokenKind::Keyword(Keyword::Table),
	UniCase::ascii("TABLES") => TokenKind::Keyword(Keyword::Tables),
	UniCase::ascii("TB") => TokenKind::Keyword(Keyword::Table),
	UniCase::ascii("TEMPFILES") => TokenKind::Keyword(Keyword::TempFiles),
	UniCase::ascii("TERMS_CACHE") => TokenKind::Keyword(Keyword::TermsCache),
	UniCase::ascii("TERMS_ORDER") => TokenKind::Keyword(Keyword::TermsOrder),
	UniCase::ascii("THEN") => TokenKind::Keyword(Keyword::Then),
	UniCase::ascii("THROW") => TokenKind::Keyword(Keyword::Throw),
	UniCase::ascii("TIMEOUT") => TokenKind::Keyword(Keyword::Timeout),
	UniCase::ascii("TO") => TokenKind::Keyword(Keyword::To),
	UniCase::ascii("TOKEN") => TokenKind::Keyword(Keyword::Token),
	UniCase::ascii("TOKENIZERS") => TokenKind::Keyword(Keyword::Tokenizers),
	UniCase::ascii("TRANSACTION") => TokenKind::Keyword(Keyword::Transaction),
	UniCase::ascii("true") => TokenKind::Keyword(Keyword::True),
	UniCase::ascii("TYPE") => TokenKind::Keyword(Keyword::Type),
	UniCase::ascii("UNIQUE") => TokenKind::Keyword(Keyword::Unique),
	UniCase::ascii("UNSET") => TokenKind::Keyword(Keyword::Unset),
	UniCase::ascii("UPDATE") => TokenKind::Keyword(Keyword::Update),
	UniCase::ascii("UPPERCASE") => TokenKind::Keyword(Keyword::Uppercase),
	UniCase::ascii("UPSERT") => TokenKind::Keyword(Keyword::Upsert),
	UniCase::ascii("URL") => TokenKind::Keyword(Keyword::Url),
	UniCase::ascii("USE") => TokenKind::Keyword(Keyword::Use),
	UniCase::ascii("USER") => TokenKind::Keyword(Keyword::User),
	UniCase::ascii("VALUE") => TokenKind::Keyword(Keyword::Value),
	UniCase::ascii("VALUES") => TokenKind::Keyword(Keyword::Values),
	UniCase::ascii("VERSION") => TokenKind::Keyword(Keyword::Version),
	UniCase::ascii("VS") => TokenKind::Keyword(Keyword::Vs),
	UniCase::ascii("WHEN") => TokenKind::Keyword(Keyword::When),
	UniCase::ascii("WHERE") => TokenKind::Keyword(Keyword::Where),
	UniCase::ascii("WITH") => TokenKind::Keyword(Keyword::With),

	// Types
	UniCase::ascii("ANY") => TokenKind::Keyword(Keyword::Any),
	UniCase::ascii("ARRAY") => TokenKind::Keyword(Keyword::Array),
	UniCase::ascii("BOOL") => TokenKind::Keyword(Keyword::Bool),
	UniCase::ascii("BYTES") => TokenKind::Keyword(Keyword::Bytes),
	UniCase::ascii("COLLECTION") => TokenKind::Keyword(Keyword::Collection),
	UniCase::ascii("DATETIME") => TokenKind::Keyword(Keyword::Datetime),
	UniCase::ascii("DECIMAL") => TokenKind::Keyword(Keyword::Decimal),
	UniCase::ascii("DURATION") => TokenKind::Keyword(Keyword::Duration),
	UniCase::ascii("FEATURE") => TokenKind::Keyword(Keyword::Feature),
	UniCase::ascii("FILE") => TokenKind::Keyword(Keyword::File),
	UniCase::ascii("FLOAT") => TokenKind::Keyword(Keyword::Float),
	UniCase::ascii("GEOMETRY") => TokenKind::Keyword(Keyword::Geometry),
	UniCase::ascii("fn") => TokenKind::Keyword(Keyword::Fn),
	UniCase::ascii("silo") => TokenKind::Keyword(Keyword::Silo),
	UniCase::ascii("mod") => TokenKind::Keyword(Keyword::Mod),
	UniCase::ascii("INT") => TokenKind::Keyword(Keyword::Int),
	UniCase::ascii("LINE") => TokenKind::Keyword(Keyword::Line),
	UniCase::ascii("MULTILINE") => TokenKind::Keyword(Keyword::MultiLine),
	UniCase::ascii("MULTIPOINT") => TokenKind::Keyword(Keyword::MultiPoint),
	UniCase::ascii("MULTIPOLYGON") => TokenKind::Keyword(Keyword::MultiPolygon),
	UniCase::ascii("NUMBER") => TokenKind::Keyword(Keyword::Number),
	UniCase::ascii("OBJECT") => TokenKind::Keyword(Keyword::Object),
	UniCase::ascii("POINT") => TokenKind::Keyword(Keyword::Point),
	UniCase::ascii("POLYGON") => TokenKind::Keyword(Keyword::Polygon),
	UniCase::ascii("RAND") => TokenKind::Keyword(Keyword::Rand),
	UniCase::ascii("RECORD") => TokenKind::Keyword(Keyword::Record),
	UniCase::ascii("REFERENCES") => TokenKind::Keyword(Keyword::References),
	UniCase::ascii("REGEX") => TokenKind::Keyword(Keyword::Regex),
	UniCase::ascii("STRING") => TokenKind::Keyword(Keyword::String),
	UniCase::ascii("ULID") => TokenKind::Keyword(Keyword::Ulid),
	UniCase::ascii("UUID") => TokenKind::Keyword(Keyword::Uuid),

	// Languages
	UniCase::ascii("AR") => TokenKind::Language(Language::Arabic),
	UniCase::ascii("ARA") => TokenKind::Language(Language::Arabic),
	UniCase::ascii("ARABIC") => TokenKind::Language(Language::Arabic),
	UniCase::ascii("DA") => TokenKind::Language(Language::Danish),
	UniCase::ascii("DAN") => TokenKind::Language(Language::Danish),
	UniCase::ascii("DANISH") => TokenKind::Language(Language::Danish),
	UniCase::ascii("DE") => TokenKind::Language(Language::German),
	UniCase::ascii("DEU") => TokenKind::Language(Language::German),
	UniCase::ascii("DUTCH") => TokenKind::Language(Language::Dutch),
	UniCase::ascii("EL") => TokenKind::Language(Language::Greek),
	UniCase::ascii("ELL") => TokenKind::Language(Language::Greek),
	UniCase::ascii("EN") => TokenKind::Language(Language::English),
	UniCase::ascii("ENG") => TokenKind::Language(Language::English),
	UniCase::ascii("ENGLISH") => TokenKind::Language(Language::English),
	UniCase::ascii("ES") => TokenKind::Language(Language::Spanish),
	UniCase::ascii("FI") => TokenKind::Language(Language::Finnish),
	UniCase::ascii("FIN") => TokenKind::Language(Language::Finnish),
	UniCase::ascii("FINNISH") => TokenKind::Language(Language::Finnish),
	UniCase::ascii("FR") => TokenKind::Language(Language::French),
	UniCase::ascii("FRA") => TokenKind::Language(Language::French),
	UniCase::ascii("FRENCH") => TokenKind::Language(Language::French),
	UniCase::ascii("GERMAN") => TokenKind::Language(Language::German),
	UniCase::ascii("GREEK") => TokenKind::Language(Language::Greek),
	UniCase::ascii("HU") => TokenKind::Language(Language::Hungarian),
	UniCase::ascii("HUN") => TokenKind::Language(Language::Hungarian),
	UniCase::ascii("HUNGARIAN") => TokenKind::Language(Language::Hungarian),
	UniCase::ascii("IT") => TokenKind::Language(Language::Italian),
	UniCase::ascii("ITA") => TokenKind::Language(Language::Italian),
	UniCase::ascii("ITALIAN") => TokenKind::Language(Language::Italian),
	UniCase::ascii("NL") => TokenKind::Language(Language::Dutch),
	UniCase::ascii("NLD") => TokenKind::Language(Language::Dutch),
	UniCase::ascii("NOR") => TokenKind::Language(Language::Norwegian),
	UniCase::ascii("NORWEGIAN") => TokenKind::Language(Language::Norwegian),
	UniCase::ascii("POR") => TokenKind::Language(Language::Portuguese),
	UniCase::ascii("PORTUGUESE") => TokenKind::Language(Language::Portuguese),
	UniCase::ascii("PT") => TokenKind::Language(Language::Portuguese),
	UniCase::ascii("RO") => TokenKind::Language(Language::Romanian),
	UniCase::ascii("ROMANIAN") => TokenKind::Language(Language::Romanian),
	UniCase::ascii("RON") => TokenKind::Language(Language::Romanian),
	UniCase::ascii("RU") => TokenKind::Language(Language::Russian),
	UniCase::ascii("RUS") => TokenKind::Language(Language::Russian),
	UniCase::ascii("RUSSIAN") => TokenKind::Language(Language::Russian),
	UniCase::ascii("SPA") => TokenKind::Language(Language::Spanish),
	UniCase::ascii("SPANISH") => TokenKind::Language(Language::Spanish),
	UniCase::ascii("SV") => TokenKind::Language(Language::Swedish),
	UniCase::ascii("SWE") => TokenKind::Language(Language::Swedish),
	UniCase::ascii("SWEDISH") => TokenKind::Language(Language::Swedish),
	UniCase::ascii("TA") => TokenKind::Language(Language::Tamil),
	UniCase::ascii("TAM") => TokenKind::Language(Language::Tamil),
	UniCase::ascii("TAMIL") => TokenKind::Language(Language::Tamil),
	UniCase::ascii("TR") => TokenKind::Language(Language::Turkish),
	UniCase::ascii("TUR") => TokenKind::Language(Language::Turkish),
	UniCase::ascii("TURKISH") => TokenKind::Language(Language::Turkish),

	// Algorithms
	UniCase::ascii("EDDSA") => TokenKind::Algorithm(Algorithm::EdDSA),
	UniCase::ascii("ES256") => TokenKind::Algorithm(Algorithm::Es256),
	UniCase::ascii("ES384") => TokenKind::Algorithm(Algorithm::Es384),
	UniCase::ascii("ES512") => TokenKind::Algorithm(Algorithm::Es512),
	UniCase::ascii("HS256") => TokenKind::Algorithm(Algorithm::Hs256),
	UniCase::ascii("HS384") => TokenKind::Algorithm(Algorithm::Hs384),
	UniCase::ascii("HS512") => TokenKind::Algorithm(Algorithm::Hs512),
	UniCase::ascii("PS256") => TokenKind::Algorithm(Algorithm::Ps256),
	UniCase::ascii("PS384") => TokenKind::Algorithm(Algorithm::Ps384),
	UniCase::ascii("PS512") => TokenKind::Algorithm(Algorithm::Ps512),
	UniCase::ascii("RS256") => TokenKind::Algorithm(Algorithm::Rs256),
	UniCase::ascii("RS384") => TokenKind::Algorithm(Algorithm::Rs384),
	UniCase::ascii("RS512") => TokenKind::Algorithm(Algorithm::Rs512),

	// Distance
	UniCase::ascii("CHEBYSHEV") => TokenKind::Distance(DistanceKind::Chebyshev),
	UniCase::ascii("COSINE") => TokenKind::Distance(DistanceKind::Cosine),
	UniCase::ascii("EUCLIDEAN") => TokenKind::Distance(DistanceKind::Euclidean),
	UniCase::ascii("HAMMING") => TokenKind::Distance(DistanceKind::Hamming),
	UniCase::ascii("JACCARD") => TokenKind::Distance(DistanceKind::Jaccard),
	UniCase::ascii("MANHATTAN") => TokenKind::Distance(DistanceKind::Manhattan),
	UniCase::ascii("MINKOWSKI") => TokenKind::Distance(DistanceKind::Minkowski),
	UniCase::ascii("PEARSON") => TokenKind::Distance(DistanceKind::Pearson),

	// VectorTypes
	UniCase::ascii("F32") => TokenKind::VectorType(VectorTypeKind::F32),
	UniCase::ascii("F64") => TokenKind::VectorType(VectorTypeKind::F64),
	UniCase::ascii("I16") => TokenKind::VectorType(VectorTypeKind::I16),
	UniCase::ascii("I32") => TokenKind::VectorType(VectorTypeKind::I32),
	UniCase::ascii("I64") => TokenKind::VectorType(VectorTypeKind::I64),

	// HTTP methods
	UniCase::ascii("GET") => TokenKind::Keyword(Keyword::Get),
	UniCase::ascii("POST") => TokenKind::Keyword(Keyword::Post),
	UniCase::ascii("PUT") => TokenKind::Keyword(Keyword::Put),
	UniCase::ascii("TRACE") => TokenKind::Keyword(Keyword::Trace)
};
