/* Copyright 2025 The OpenXLA Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include <cassert>
#include <memory>
#include <utility>

#include "mlir/Dialect/Func/IR/FuncOps.h"
#include "mlir/Dialect/LLVMIR/LLVMDialect.h"  // IWYU pragma: keep
#include "mlir/Dialect/Tensor/IR/Tensor.h"
#include "mlir/Dialect/Vector/IR/VectorOps.h"
#include "mlir/IR/AffineExpr.h"
#include "mlir/IR/MLIRContext.h"
#include "mlir/IR/PatternMatch.h"
#include "mlir/Pass/Pass.h"
#include "mlir/Support/LLVM.h"
#include "mlir/Transforms/GreedyPatternRewriteDriver.h"
#include "xla/backends/cpu/codegen/tiled/transforms/passes.h"

namespace xla::cpu {

#define GEN_PASS_DECL_TENSOROPSTOBUFFERIZABLEPASS
#define GEN_PASS_DEF_TENSOROPSTOBUFFERIZABLEPASS
#include "xla/backends/cpu/codegen/tiled/transforms/passes.h.inc"

namespace {

struct TensorToArithBitcast : mlir::OpRewritePattern<mlir::tensor::BitcastOp> {
  using OpRewritePattern::OpRewritePattern;

  mlir::LogicalResult matchAndRewrite(
      mlir::tensor::BitcastOp op,
      mlir::PatternRewriter& rewriter) const override {
    rewriter.replaceOpWithNewOp<mlir::arith::BitcastOp>(op, op.getType(),
                                                        op.getOperand());
    return mlir::success();
  }
};

class TensorOpsToBufferizablePass
    : public impl::TensorOpsToBufferizablePassBase<
          TensorOpsToBufferizablePass> {
 public:
  using TensorOpsToBufferizablePassBase::TensorOpsToBufferizablePassBase;

  void runOnOperation() override {
    mlir::MLIRContext* context = &getContext();
    mlir::RewritePatternSet patterns(context);
    patterns.add<TensorToArithBitcast>(context);
    if (mlir::failed(
            mlir::applyPatternsGreedily(getOperation(), std::move(patterns)))) {
      signalPassFailure();
      return;
    }
  }
};

}  // namespace

std::unique_ptr<mlir::Pass> CreateTensorOpsToBufferizablePass() {
  return std::make_unique<TensorOpsToBufferizablePass>();
}

}  // namespace xla::cpu
