/* Copyright 2019 The OpenXLA Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#ifndef XLA_BACKENDS_GPU_RUNTIME_TRIANGULAR_SOLVE_THUNK_H_
#define XLA_BACKENDS_GPU_RUNTIME_TRIANGULAR_SOLVE_THUNK_H_

#include <cstdint>
#include <functional>
#include <memory>
#include <numeric>

#include "absl/status/status.h"
#include "absl/status/statusor.h"
#include "absl/types/span.h"
#include "xla/backends/gpu/runtime/shaped_slice.h"
#include "xla/backends/gpu/runtime/thunk.h"
#include "xla/backends/gpu/runtime/thunk.pb.h"
#include "xla/runtime/buffer_use.h"
#include "xla/service/buffer_assignment.h"
#include "xla/shape_util.h"
#include "xla/stream_executor/blas.h"
#include "xla/stream_executor/device_address.h"
#include "xla/stream_executor/stream.h"
#include "xla/xla_data.pb.h"

namespace xla {
namespace gpu {

// This class stores everything that StreamExecutor needs to launch a triangular
// solve (BlasTrsm). It is generated by IrEmitter.
//
// Thread-compatible.
class TriangularSolveThunk : public Thunk {
 public:
  TriangularSolveThunk(ThunkInfo thunk_info,
                       const TriangularSolveOptions& options,
                       const ShapedSlice& a_buffer, const ShapedSlice& b_buffer,
                       const ShapedSlice& temp_buffer);

  TriangularSolveThunk(const TriangularSolveThunk&) = delete;
  TriangularSolveThunk& operator=(const TriangularSolveThunk&) = delete;

  absl::Status ExecuteOnStream(const ExecuteParams& params) override;

  BufferUses buffer_uses() const override {
    return {
        BufferUse::Read(a_buffer_.slice, a_buffer_.shape),
        BufferUse::Write(b_buffer_.slice, b_buffer_.shape),
        BufferUse::Scratch(temp_buffer_.slice, temp_buffer_.shape),
    };
  };

  static absl::StatusOr<std::unique_ptr<TriangularSolveThunk>> FromProto(
      ThunkInfo thunk_info, const TriangularSolveThunkProto& proto,
      absl::Span<const BufferAllocation> allocations);

  absl::StatusOr<ThunkProto> ToProto() const override;

 private:
  int64_t batch_size() const {
    return std::accumulate(b_buffer_.shape.dimensions().begin(),
                           b_buffer_.shape.dimensions().end() - 2, int64_t{1},
                           std::multiplies<int64_t>());
  }

  int64_t a_batch_stride() const {
    int64_t elem_size = ShapeUtil::ByteSizeOfPrimitiveType(type_);
    return side_ == se::blas::Side::kLeft ? (m_ * m_ * elem_size)
                                          : (n_ * n_ * elem_size);
  }

  int64_t b_batch_stride() const {
    return m_ * n_ * ShapeUtil::ByteSizeOfPrimitiveType(type_);
  }

  const se::blas::UpperLower uplo_;
  const se::blas::Side side_;
  const se::blas::Diagonal unit_diagonal_;
  se::blas::Transpose transpose_a_;

  const ShapedSlice a_buffer_;
  const ShapedSlice b_buffer_;
  const ShapedSlice temp_buffer_;

  const PrimitiveType type_;
  const int64_t m_;
  const int64_t n_;
};

absl::Status RunTriangularSolve(se::DeviceAddressBase a_data,
                                se::DeviceAddressBase b_data,
                                se::DeviceAddressBase temp_data,
                                se::blas::UpperLower uplo, se::blas::Side side,
                                se::blas::Diagonal unit_diagonal,
                                se::blas::Transpose transpose_a,
                                PrimitiveType type, int64_t batch_size,
                                int64_t m, int64_t n, int64_t a_batch_stride,
                                int64_t b_batch_stride, se::Stream* stream);

}  // namespace gpu
}  // namespace xla

#endif  // XLA_BACKENDS_GPU_RUNTIME_TRIANGULAR_SOLVE_THUNK_H_
