/* Copyright 2018 The OpenXLA Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "xla/hlo/transforms/simplifiers/bfloat16_conversion_folding.h"

#include <cstdint>
#include <vector>

#include "absl/container/flat_hash_set.h"
#include "absl/log/check.h"
#include "absl/status/status.h"
#include "absl/status/statusor.h"
#include "absl/strings/string_view.h"
#include "xla/hlo/analysis/alias_info.h"
#include "xla/hlo/ir/dfs_hlo_visitor_with_default.h"
#include "xla/hlo/ir/hlo_computation.h"
#include "xla/hlo/ir/hlo_instruction.h"
#include "xla/hlo/ir/hlo_opcode.h"
#include "xla/service/float_support.h"
#include "xla/shape_util.h"
#include "xla/tsl/platform/errors.h"
#include "xla/util.h"
#include "xla/xla_data.pb.h"

namespace xla {

class BFloat16ConversionFoldingVisitor : public DfsHloVisitorWithDefault {
 public:
  explicit BFloat16ConversionFoldingVisitor(
      HloComputation* computation, const FloatSupport* bfloat16_support,
      const AliasInfo* alias_info,
      BFloat16ConversionFolding* bfloat16_conversion_folding)
      : computation_(computation),
        bfloat16_support_(bfloat16_support),
        alias_info_(alias_info),
        bfloat16_conversion_folding_(bfloat16_conversion_folding) {}

  absl::Status DefaultAction(HloInstruction* hlo) override;

  // Special handling for all-reduce which can have a tuple output.
  absl::Status HandleAllReduce(HloInstruction* crs) override;

  static bool Run(HloComputation* computation,
                  const FloatSupport* bfloat16_support,
                  const AliasInfo* alias_info,
                  BFloat16ConversionFolding* bfloat16_conversion_folding) {
    BFloat16ConversionFoldingVisitor visitor(
        computation, bfloat16_support, alias_info, bfloat16_conversion_folding);
    CHECK_OK(computation->Accept(&visitor));
    return visitor.changed_;
  }

 private:
  // Checks if the HLO has a BF16 -> F32 conversion as input, or a F32 -> BF16
  // conversion as output, and folds them to the HLO itself if feasible.
  absl::Status TryFoldBF16Conversions(HloInstruction* hlo);

  // Folds the F32 -> BF16 conversions from the HLO's output.
  //
  // Precondition: all of the HLO's users are F32 -> BF16 conversions.
  absl::Status FoldOutputConversions(HloInstruction* hlo);

  // Folds the BF16 -> F32 conversion operand to the HLO.
  //
  // Precondition: the operand is a BF16 -> F32 conversion.
  absl::Status FoldOperandConversion(HloInstruction* hlo,
                                     int64_t operand_index);

  HloComputation* computation_;
  const FloatSupport* bfloat16_support_;
  const AliasInfo* alias_info_;
  BFloat16ConversionFolding* bfloat16_conversion_folding_;
  bool changed_ = false;
};

absl::Status BFloat16ConversionFoldingVisitor::FoldOutputConversions(
    HloInstruction* hlo) {
  std::vector<HloInstruction*> materialized_users = hlo->users();
  hlo->mutable_shape()->set_element_type(BF16);
  bfloat16_conversion_folding_->UpdateLayout(hlo->mutable_shape());
  for (auto user : materialized_users) {
    CHECK_EQ(user->opcode(), HloOpcode::kConvert);
    TF_RETURN_IF_ERROR(user->ReplaceAllUsesWith(hlo));
    changed_ = true;
  }
  return absl::OkStatus();
}

absl::Status BFloat16ConversionFoldingVisitor::FoldOperandConversion(
    HloInstruction* hlo, int64_t operand_index) {
  // The operand is a convert from BF16 to F32.
  auto operand = hlo->mutable_operand(operand_index);
  CHECK_EQ(operand->opcode(), HloOpcode::kConvert);
  TF_RETURN_IF_ERROR(
      hlo->ReplaceOperandWith(operand_index, operand->mutable_operand(0)));
  changed_ = true;
  return absl::OkStatus();
}

namespace {

// Returns whether hlo has users and all users are conversions from F32 to BF16.
bool AllUsersAreF32ToBF16Converts(const HloInstruction* hlo) {
  if (hlo->user_count() == 0 || hlo->shape().element_type() != F32) {
    return false;
  }
  for (const auto user : hlo->users()) {
    if (user->opcode() == HloOpcode::kConvert &&
        user->shape().element_type() == BF16) {
      continue;
    }
    return false;
  }
  return true;
}

}  // namespace

absl::Status BFloat16ConversionFoldingVisitor::TryFoldBF16Conversions(
    HloInstruction* hlo) {
  std::vector<int64_t> bf16_to_f32_operands;
  bool has_other_f32_operands = false;
  for (int64_t i = 0; i < hlo->operands().size(); ++i) {
    auto operand = hlo->operand(i);
    if (operand->shape().element_type() == F32) {
      if (operand->opcode() == HloOpcode::kConvert &&
          operand->operand(0)->shape().element_type() == BF16 &&
          bfloat16_support_->SupportsLowPrecisionOperand(*hlo, i)) {
        // Operand is a convert from BF16 to F32 and we support BF16 input
        // directly in the current HLO at the operand index.
        bf16_to_f32_operands.push_back(i);
      } else {
        has_other_f32_operands = true;
      }
      continue;
    }
  }

  const bool fold_output_conversion =
      AllUsersAreF32ToBF16Converts(hlo) &&
      bfloat16_support_->SupportsLowPrecisionOutput(*hlo);

  if (!bfloat16_support_->SupportsMixedPrecisions(*hlo)) {
    if (has_other_f32_operands ||
        (!fold_output_conversion && hlo->shape().element_type() == F32)) {
      // Some of the operands/output will remain F32, but we cannot use mixed
      // precisions, so we cannot do anything here.
      return absl::OkStatus();
    }
  }

  if (fold_output_conversion) {
    TF_RETURN_IF_ERROR(FoldOutputConversions(hlo));
  }

  for (int64_t i : bf16_to_f32_operands) {
    TF_RETURN_IF_ERROR(FoldOperandConversion(hlo, i));
  }
  return absl::OkStatus();
}

absl::Status BFloat16ConversionFoldingVisitor::DefaultAction(
    HloInstruction* hlo) {
  // Do not fold BF16 conversions for instructions related to tuples, entry and
  // exit of a computation, fusion, convert, side-effecting instructions,
  // in-place operations and control flow.
  if (hlo->opcode() == HloOpcode::kTuple ||                     //
      hlo->opcode() == HloOpcode::kGetTupleElement ||           //
      hlo->opcode() == HloOpcode::kConstant ||                  //
      hlo->opcode() == HloOpcode::kParameter ||                 //
      hlo->opcode() == HloOpcode::kFusion ||                    //
      hlo->opcode() == HloOpcode::kBitcast ||                   //
      hlo->opcode() == HloOpcode::kBitcastConvert ||            //
      hlo->opcode() == HloOpcode::kConvert ||                   //
      hlo->opcode() == HloOpcode::kCall ||                      //
      hlo->opcode() == HloOpcode::kCustomCall ||                //
      hlo->opcode() == HloOpcode::kWhile ||                     //
      hlo->opcode() == HloOpcode::kConditional ||               //
      hlo->opcode() == HloOpcode::kAsyncStart ||                //
      hlo->opcode() == HloOpcode::kAsyncDone ||                 //
      hlo->opcode() == HloOpcode::kOptimizationBarrier ||       //
      !alias_info_->GetInPlaceInputOutputPairs(hlo).empty() ||  //
      hlo->HasSideEffectNoRecurse()) {
    return absl::OkStatus();
  }
  if (hlo == computation_->root_instruction() &&
      !bfloat16_support_->SupportsMixedPrecisions(*hlo)) {
    // If hlo is the root instruction, we cannot change its output, so folding
    // can only happen when it supports mixed precision so that we can change
    // its operands.
    return absl::OkStatus();
  }
  return TryFoldBF16Conversions(hlo);
}

absl::Status BFloat16ConversionFoldingVisitor::HandleAllReduce(
    HloInstruction* crs) {
  if (crs->HasSideEffectNoRecurse()) {
    // Do not perform optimization on side-effected AllReduce.
    return absl::OkStatus();
  }
  // First use DefaultAction() to handle the operands. It can't handle
  // tuple-shaped output.
  TF_RETURN_IF_ERROR(DefaultAction(crs));

  if (!bfloat16_support_->SupportsMixedPrecisions(*crs)) {
    return absl::OkStatus();
  }

  // If the output is not a tuple, we don't need special handling.
  if (!crs->shape().IsTuple()) {
    return absl::OkStatus();
  }

  // If crs is the root instruction, we should keep its original output type.
  // The root instruction implicitly has a use from being the result of the
  // computation, and the code below does not take this use into account.
  if (crs == computation_->root_instruction()) {
    return absl::OkStatus();
  }

  // Then do per-tuple-element handling on the output.
  std::vector<std::vector<HloInstruction*>> per_tuple_element_gtes(
      crs->operand_count());
  for (auto user : crs->users()) {
    if (user->opcode() != HloOpcode::kGetTupleElement) {
      return absl::OkStatus();
    }
    per_tuple_element_gtes[user->tuple_index()].push_back(user);
  }

  for (int64_t i = 0; i < crs->operand_count(); ++i) {
    // Fold conversions only when all the get-tuple-elements' users are
    // conversions from F32 to BF16.
    auto all_gte_users_are_bf16_convert = [&per_tuple_element_gtes, i]() {
      // If no uses then return false. (As no uses are bf16 converts).
      if (per_tuple_element_gtes[i].empty()) {
        return false;
      }
      for (auto gte : per_tuple_element_gtes[i]) {
        if (!AllUsersAreF32ToBF16Converts(gte)) {
          return false;
        }
      }
      return true;
    };
    if (!all_gte_users_are_bf16_convert()) {
      continue;
    }

    ShapeUtil::GetMutableSubshape(crs->mutable_shape(), {i})
        ->set_element_type(BF16);
    bfloat16_conversion_folding_->UpdateLayout(
        ShapeUtil::GetMutableSubshape(crs->mutable_shape(), {i}));
    for (auto gte : per_tuple_element_gtes[i]) {
      TF_RETURN_IF_ERROR(FoldOutputConversions(gte));
    }
  }

  return absl::OkStatus();
}

absl::StatusOr<bool> BFloat16ConversionFolding::RunImpl(
    HloModule* module,
    const absl::flat_hash_set<absl::string_view>& execution_threads) {
  XLA_VLOG_LINES(2, "BFloat16ConversionFolding::RunImpl(), before:\n" +
                        module->ToString());
  bool changed = false;
  for (auto* comp : module->MakeNonfusionComputations(execution_threads)) {
    if (BFloat16ConversionFoldingVisitor::Run(comp, bfloat16_support_,
                                              alias_info_, this)) {
      changed = true;
    }
  }
  XLA_VLOG_LINES(
      2, "BFloat16ConversionFolding::RunImpl(), after:\n" + module->ToString());
  return changed;
}

}  // namespace xla
