/* Copyright 2017 The OpenXLA Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "xla/service/cpu/cpu_layout_assignment.h"

#include <cstdint>
#include <numeric>
#include <optional>
#include <vector>

#include "absl/algorithm/container.h"
#include "absl/container/flat_hash_map.h"
#include "absl/log/check.h"
#include "absl/status/status.h"
#include "xla/backends/cpu/codegen/target_machine_features.h"
#include "xla/hlo/ir/hlo_casting_utils.h"
#include "xla/hlo/ir/hlo_instruction.h"
#include "xla/hlo/ir/hlo_instructions.h"
#include "xla/hlo/ir/hlo_opcode.h"
#include "xla/layout_util.h"
#include "xla/service/cpu/dot_op_emitter.h"
#include "xla/service/cpu/ir_emission_utils.h"
#include "xla/shape.h"
#include "xla/shape_util.h"
#include "xla/tsl/platform/errors.h"

namespace xla {
namespace cpu {

// We want to change the layout of constant arrays to be column major when all
// of their users are dot operations that can be made faster with the flipped
// layout.  To avoid going quadratic over the # of instructions, we cache this
// property in should_make_rhs_col_major -- it maps a constant to true if all of
// the users of said constant are dot operations that can be sped up.  This
// cache is populated lazily as we encounter dot operations traversing the
// instruction stream.

namespace {
using std::nullopt;
using std::optional;

using ShouldMakeOperandColMajorCache =
    absl::flat_hash_map<const HloInstruction*, bool>;
}  // namespace

static bool ShouldMakeAllUsersColMajor(const HloInstruction* instruction) {
  for (auto* user : instruction->users()) {
    optional<int64_t> operand_idx =
        ProfitableToMakeDotOperandColumnMajor(*user);
    if (!operand_idx || user->operand(*operand_idx) != instruction ||
        absl::c_count(user->operands(), instruction) != 1) {
      return false;
    }
  }
  return true;
}

static optional<int64_t> ShouldMakeOperandColumnMajor(
    ShouldMakeOperandColMajorCache* cache, const HloInstruction& instruction) {
  optional<int64_t> operand_idx =
      ProfitableToMakeDotOperandColumnMajor(instruction);
  if (!operand_idx) {
    return nullopt;
  }

  const HloInstruction* operand = instruction.operand(*operand_idx);
  if (operand->opcode() != HloOpcode::kConstant) {
    return nullopt;
  }

  auto it = cache->find(operand);
  if (it == cache->end()) {
    auto insert_result =
        cache->insert({operand, ShouldMakeAllUsersColMajor(operand)});
    CHECK(insert_result.second);
    it = insert_result.first;
  }

  return it->second ? operand_idx : nullopt;
}

static Shape RowMajorShape(Shape shape) {
  ShapeUtil::ForEachMutableSubshape(
      &shape, [](Shape* subshape, const ShapeIndex& index) {
        if (!subshape->IsArray()) {
          return;
        }
        std::vector<int64_t> dimension_order(subshape->dimensions().size());
        std::iota(dimension_order.rbegin(), dimension_order.rend(), 0);
        *subshape->mutable_layout() = LayoutUtil::MakeLayout(dimension_order);
      });
  return shape;
}

static Shape ColMajorShape(const Shape& old_shape) {
  Shape new_shape(old_shape);
  std::vector<int64_t> dimension_order(new_shape.dimensions().size());
  absl::c_iota(dimension_order, 0);
  *new_shape.mutable_layout() = LayoutUtil::MakeLayout(dimension_order);
  return new_shape;
}

static bool OperandsAndResultMustHaveRowMajorLayout(
    const HloInstruction& instr,
    const TargetMachineFeatures& target_machine_features) {
  if (instr.opcode() == HloOpcode::kConvolution) {
    return PotentiallyImplementedAsEigenConvolution(instr,
                                                    target_machine_features);
  }
  if (instr.opcode() == HloOpcode::kDot) {
    return DotOperandsAndResultMustHaveRowMajorLayout(
        instr, target_machine_features, /*allow_runtime_calls=*/true);
  }
  if (instr.opcode() == HloOpcode::kCustomCall) {
    return instr.custom_call_target() == "TopK";
  }
  return false;
}

absl::Status CpuLayoutAssignment::AddBackendConstraints(
    LayoutConstraints* constraints) {
  ShouldMakeOperandColMajorCache cache;

  const HloComputation* computation = constraints->computation();
  for (auto* instruction : computation->instructions()) {
    if (OperandsAndResultMustHaveRowMajorLayout(*instruction,
                                                target_machine_features_)) {
      TF_RETURN_IF_ERROR(SetInstructionLayout(
          RowMajorShape(instruction->shape()), instruction));
      for (int i = 0; i < instruction->operand_count(); i++) {
        TF_RETURN_IF_ERROR(SetOperandLayout(
            RowMajorShape(instruction->operand(i)->shape()), instruction, i));
      }
    } else if (optional<int64_t> op_idx =
                   ShouldMakeOperandColumnMajor(&cache, *instruction)) {
      const HloInstruction* op = instruction->operand(*op_idx);
      TF_RETURN_IF_ERROR(
          SetOperandLayout(ColMajorShape(op->shape()), instruction, *op_idx));
    } else if (instruction->opcode() == HloOpcode::kReduceScatter) {
      // XLA:CPU can only support reduce-scatter where the scatter dimension
      // is the most major dimension in the layout.
      auto ars = Cast<HloReduceScatterInstruction>(instruction);
      TF_RETURN_IF_ERROR(SetInstructionLayout(
          ShapeUtil::MoveDimToMajor(ars->shape(), ars->scatter_dimension()),
          ars));
    } else if (instruction->opcode() == HloOpcode::kAllGather) {
      // XLA:CPU can only support all-gathers where the gather dimension is the
      // most major dimension in the layout.
      auto ag = Cast<HloAllGatherInstruction>(instruction);
      TF_RETURN_IF_ERROR(SetInstructionLayout(
          ShapeUtil::MoveDimToMajor(ag->shape(), ag->all_gather_dimension()),
          ag));
    } else {
      for (int64_t operand_no = 0; operand_no < instruction->operand_count();
           ++operand_no) {
        // Skip operands which already have a constraint.
        if (constraints->OperandLayout(instruction, operand_no) != nullptr) {
          continue;
        }
        // Skip over forwarded operands.
        if (AnyOperandBufferForwarded(instruction, operand_no)) {
          continue;
        }
        // Skip operands with non-array shapes.
        if (!instruction->operand(operand_no)->shape().IsArray()) {
          continue;
        }
        Shape operand_shape(
            RowMajorShape(instruction->operand(operand_no)->shape()));
        TF_RETURN_IF_ERROR(
            SetOperandLayout(operand_shape, instruction, operand_no));
      }
      // Skip over the root instruction for the top-level computation.
      if (computation->parent()->entry_computation() == computation &&
          computation->root_instruction() == instruction) {
        continue;
      }
      // Skip instructions which don't produce array shapes (tuples, opaque,
      // etc.).
      if (!instruction->shape().IsArray()) {
        continue;
      }
    }
  }
  return absl::OkStatus();
}
}  // namespace cpu
}  // namespace xla
