from django.contrib import admin, messages
from django.contrib.auth.admin import GroupAdmin as BaseGroupAdmin
from django.contrib.auth.admin import UserAdmin as BaseUserAdmin
from django.contrib.auth.models import Group
from django.core.validators import EMPTY_VALUES
from django.shortcuts import redirect
from django.urls import reverse_lazy
from django.utils.translation import gettext_lazy as _

from example.models import (
    ActionUser,
    ApprovalChoices,
    Category,
    ColorChoices,
    FilterUser,
    Label,
    Post,
    PriorityChoices,
    Project,
    SectionUser,
    StatusChoices,
    Tag,
    Task,
    User,
)
from unfold.admin import ModelAdmin, StackedInline
from unfold.contrib.filters.admin import (
    AllValuesCheckboxFilter,
    AutocompleteSelectFilter,
    AutocompleteSelectMultipleFilter,
    BooleanRadioFilter,
    CheckboxFilter,
    ChoicesCheckboxFilter,
    ChoicesDropdownFilter,
    ChoicesRadioFilter,
    DropdownFilter,
    FieldTextFilter,
    MultipleChoicesDropdownFilter,
    MultipleDropdownFilter,
    MultipleRelatedDropdownFilter,
    RadioFilter,
    RangeDateFilter,
    RangeDateTimeFilter,
    RangeNumericFilter,
    RangeNumericListFilter,
    RelatedCheckboxFilter,
    RelatedDropdownFilter,
    SingleNumericFilter,
    SliderNumericFilter,
    TextFilter,
)
from unfold.datasets import BaseDataset
from unfold.decorators import action, display
from unfold.forms import AdminPasswordChangeForm, UserChangeForm, UserCreationForm
from unfold.sections import TableSection, TemplateSection
from unfold.widgets import UnfoldAdminCheckboxSelectMultiple, UnfoldAdminSelect2Widget

admin.site.unregister(Group)


class UserTagInline(StackedInline):
    model = User.tags.through
    per_page = 1
    collapsible = True
    tab = True


class PostInline(StackedInline):
    model = Post
    ordering_field = "weight"
    hide_ordering_field = True
    list_display = ["title", "weight"]


class ProjectDatasetModelAdmin(ModelAdmin):
    pass


class ProjectDataset(BaseDataset):
    model = Project
    model_admin = ProjectDatasetModelAdmin
    tab = True


class ExtendedUserChangeForm(UserChangeForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields["status"].widget = UnfoldAdminSelect2Widget(choices=StatusChoices)
        self.fields["projects"].widget = UnfoldAdminCheckboxSelectMultiple(
            choices=Project.objects.all().values_list("id", "name")
        )


@admin.register(User)
class UserAdmin(BaseUserAdmin, ModelAdmin):
    form = ExtendedUserChangeForm
    add_form = UserCreationForm
    change_password_form = AdminPasswordChangeForm
    inlines = [UserTagInline, PostInline]
    change_form_datasets = [
        ProjectDataset,
    ]
    autocomplete_fields = ["tags"]
    compressed_fields = True
    readonly_fields = ["custom_readonly_field"]
    fieldsets = (
        (
            None,
            {
                "fields": (
                    # "username",
                    "password",
                    "custom_readonly_field",
                )
            },
        ),
        (
            _("Personal info"),
            {
                "fields": (
                    ("first_name", "last_name"),
                    "email",
                    "status",
                    "tags",
                    "projects",
                    (),
                ),
                "classes": ["tab"],
            },
        ),
        (
            _("Permissions"),
            {
                "fields": (
                    "username",  # Test the error count tab
                    "is_active",
                    "is_staff",
                    "is_superuser",
                    "groups",
                    "user_permissions",
                ),
                "classes": ["tab"],
            },
        ),
        (_("Important dates"), {"fields": ("last_login", "date_joined")}),
    )

    @display(description="Custom readonly field")
    def custom_readonly_field(self, obj):
        return "Custom readonly field"


class RelatedTableSection(TableSection):
    verbose_name = "Related log entries"
    related_name = "logentry_set"
    columns = [
        "object_id",
    ]


class TagSection(TableSection):
    related_name = "tags"
    fields = ["name"]


class SomeTemplateSection(TemplateSection):
    template_name = "section_template.html"


@admin.register(SectionUser)
class SectionUserAdmin(UserAdmin):
    form = UserChangeForm
    add_form = UserCreationForm
    change_password_form = AdminPasswordChangeForm
    list_sections = [
        SomeTemplateSection,
        RelatedTableSection,
    ]


class CustomTextFilter(TextFilter):
    title = _("Text filter")
    parameter_name = "text_username"

    def queryset(self, request, queryset):
        if self.value() not in EMPTY_VALUES:
            return queryset.filter(username__icontains=self.value())

        return queryset


class CustomRangeNumericListFilter(RangeNumericListFilter):
    parameter_name = "numeric_range_custom"
    title = "Numeric Range Custom"


class CustomSliderNumericFilter(SliderNumericFilter):
    MAX_DECIMALS = 2
    STEP = 1


class CustomStatusRadioFilter(RadioFilter):
    title = _("Custom radio filter")
    parameter_name = "custom_radio_filter"

    def lookups(self, request, model_admin):
        return StatusChoices.choices

    def queryset(self, request, queryset):
        if self.value() not in EMPTY_VALUES:
            return queryset.filter(status=self.value())

        return queryset


class CustomApprovalCheckboxFilter(CheckboxFilter):
    title = _("Custom checkbox filter")
    parameter_name = "custom_checkbox_filter"

    def lookups(self, request, model_admin):
        return ApprovalChoices.choices

    def queryset(self, request, queryset):
        if self.value() not in EMPTY_VALUES:
            return queryset.filter(approval__in=self.value())

        return queryset


class CustomPriorityDropdownFilter(DropdownFilter):
    title = _("Custom priority dropdown filter")
    parameter_name = "custom_priority"

    def lookups(self, request, model_admin):
        return PriorityChoices.choices

    def queryset(self, request, queryset):
        if self.value() not in EMPTY_VALUES:
            return queryset.filter(priority=self.value())

        return queryset


class CustomColorMultipleDropdownFilter(MultipleDropdownFilter):
    title = _("Custom color multiple dropdown filter")
    parameter_name = "custom_color"

    def lookups(self, request, model_admin):
        return ColorChoices.choices

    def queryset(self, request, queryset):
        if self.value() not in EMPTY_VALUES:
            return queryset.filter(color__in=self.value())

        return queryset


@admin.register(FilterUser)
class FilterUserAdmin(UserAdmin):
    list_fullwidth = True
    list_display = [
        "username",
        "email",
        "is_active",
        "is_staff",
        "is_active",
        "status",
        "approval",
        "date_joined",
        "last_login",
    ]
    list_filter = [
        CustomTextFilter,
        ("username", FieldTextFilter),
        # Autocomplete filters
        ("projects", AutocompleteSelectFilter),
        ("tasks", AutocompleteSelectMultipleFilter),
        # Dropdown filters
        ("priority", ChoicesDropdownFilter),
        ("color", MultipleChoicesDropdownFilter),
        ("categories", RelatedDropdownFilter),
        ("labels", MultipleRelatedDropdownFilter),
        CustomPriorityDropdownFilter,
        CustomColorMultipleDropdownFilter,
        # Date/time filters
        ("date_joined", RangeDateFilter),
        ("last_login", RangeDateTimeFilter),
        # Choice filters
        ("status", ChoicesRadioFilter),
        ("approval", ChoicesCheckboxFilter),
        ("is_active", BooleanRadioFilter),
        ("tags", RelatedCheckboxFilter),
        ("username", AllValuesCheckboxFilter),
        CustomStatusRadioFilter,
        CustomApprovalCheckboxFilter,
        # Numeric filters
        ("numeric_single", SingleNumericFilter),
        ("numeric_slider", SliderNumericFilter),
        ("numeric_slider_custom", CustomSliderNumericFilter),
        ("numeric_range", RangeNumericFilter),
        CustomRangeNumericListFilter,
    ]
    list_filter_submit = True
    list_filter_sheet = False
    fieldsets = [
        (
            None,
            {
                "fields": (
                    "username",
                    "status",
                    "approval",
                    "numeric_single",
                    "numeric_slider",
                    "numeric_slider_custom",
                    "numeric_range",
                    "numeric_range_custom",
                ),
            },
        ),
    ]


@admin.register(ActionUser)
class ActionsUserAdmin(BaseUserAdmin, ModelAdmin):
    form = UserChangeForm
    add_form = UserCreationForm
    change_password_form = AdminPasswordChangeForm
    actions_list = [
        "changelist_action",
        "changelist_action_mixed_permissions_true",
        "changelist_action_mixed_permissions_false",
        "changelist_action_mixed_permissions_perm_not_granted",
        "changelist_action_permission_true",
        "changelist_action_permission_false",
        "changelist_action_multiple_different_permissions",
        {
            "title": "Changelist dropdown for actions",
            "items": [
                "changelist_action_dropdown",
            ],
        },
    ]
    actions_row = [
        "changelist_row_action",
        "changelist_row_action_mixed_permissions_true",
        "changelist_row_action_mixed_permissions_false",
        "changelist_row_action_mixed_permissions_perm_not_granted",
        "changelist_row_action_permission_true",
        "changelist_row_action_permission_false",
        "changelist_row_action_multiple_different_permissions",
    ]
    actions_detail = [
        "changeform_action",
        "changeform_action_mixed_permissions_true",
        "changeform_action_mixed_permissions_false",
        "changeform_action_mixed_permissions_perm_not_granted",
        "changeform_action_permission_true",
        "changeform_action_permission_false",
        "changeform_action_multiple_different_permissions",
        {
            "title": "Changeform dropdown for actions",
            "items": [
                "changeform_action_dropdown",
            ],
        },
    ]
    actions_submit_line = [
        "submit_line_action",
        "submit_line_action_mixed_permissions_true",
        "submit_line_action_mixed_permissions_false",
        "submit_line_action_mixed_permissions_perm_not_granted",
        "submit_line_action_permission_true",
        "submit_line_action_permission_false",
        "submit_line_action_multiple_different_permissions",
    ]

    ######################################################################
    # Changelist actions
    ######################################################################
    @action(description="Changelist action dropdown")
    def changelist_action_dropdown(self, request):
        messages.success(request, "Changelist action dropdown successfully executed")
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(description="Changelist action")
    def changelist_action(self, request):
        messages.success(request, "Changelist action successfully executed")
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist action with mixed permissions true",
        permissions=["changelist_action_true", "example.view_user"],
    )
    def changelist_action_mixed_permissions_true(self, request):
        messages.success(
            request,
            "Changelist action with mixed permissions true successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist action with mixed permissions false",
        permissions=["changelist_action_false", "example.view_user"],
    )
    def changelist_action_mixed_permissions_false(self, request):
        messages.success(
            request,
            "Changelist action with mixed permissions false successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist action with mixed permissions perm not granted",
        permissions=["changelist_action_true", "example.delete_user"],
    )
    def changelist_action_mixed_permissions_perm_not_granted(self, request):
        messages.success(
            request,
            "Changelist action with mixed permissions perm not granted successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist action permission true",
        permissions=["changelist_action_true"],
    )
    def changelist_action_permission_true(self, request):
        messages.success(
            request, "Changelist action with true permission successfully executed"
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist action permission false",
        permissions=["changelist_action_false"],
    )
    def changelist_action_permission_false(self, request):
        messages.success(
            request, "Changelist action with false permission successfully executed"
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist action with multiple permissions",
        permissions=[
            "changelist_action_true",
            "changelist_action_false",
        ],
    )
    def changelist_action_multiple_different_permissions(self, request):
        messages.success(
            request,
            "Changelist action with multiple different permissions successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    def has_changelist_action_true_permission(self, request):
        return True

    def has_changelist_action_false_permission(self, request):
        return False

    ######################################################################
    # Changelist row actions
    ######################################################################
    @action(description="Changelist row action")
    def changelist_row_action(self, request, object_id):
        messages.success(request, "Changelist row action successfully executed")
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist row action with mixed permissions true",
        permissions=["changelist_row_action_true", "example.view_user"],
    )
    def changelist_row_action_mixed_permissions_true(self, request, object_id):
        messages.success(
            request,
            "Changelist row action with mixed permissions true successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist row action with mixed permissions false",
        permissions=["changelist_row_action_false", "example.view_user"],
    )
    def changelist_row_action_mixed_permissions_false(self, request, object_id):
        messages.success(
            request,
            "Changelist row action with mixed permissions false successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist row action with mixed permissions perm not granted",
        permissions=["changelist_row_action_true", "example.delete_user"],
    )
    def changelist_row_action_mixed_permissions_perm_not_granted(
        self, request, object_id
    ):
        messages.success(
            request,
            "Changelist row action with mixed permissions perm not granted successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist row action permission true",
        permissions=["changelist_row_action_true"],
    )
    def changelist_row_action_permission_true(self, request, object_id):
        messages.success(
            request,
            "Changelist row action with true permission successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist row action permission false",
        permissions=["changelist_row_action_false"],
    )
    def changelist_row_action_permission_false(self, request, object_id):
        messages.success(
            request,
            "Changelist row action with false permission successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changelist row action with multiple permissions",
        permissions=[
            "changelist_row_action_true",
            "changelist_row_action_false",
        ],
    )
    def changelist_row_action_multiple_different_permissions(self, request, object_id):
        messages.success(
            request,
            "Changelist row action with multiple different permissions successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    def has_changelist_row_action_true_permission(self, request):
        return True

    def has_changelist_row_action_false_permission(self, request):
        return False

    ######################################################################
    # Changeform actions
    ######################################################################
    @action(
        description="Changeform action dropdown",
        permissions=["changeform_action_dropdown"],
    )
    def changeform_action_dropdown(self, request, object_id):
        messages.success(request, "Changeform action dropdown successfully executed")
        return redirect(reverse_lazy("admin:example_user_changelist"))

    def has_changeform_action_dropdown_permission(self, request, object_id):
        return True

    @action(
        description="Changeform action with mixed permissions true",
        permissions=["example.view_user", "changeform_action_true"],
    )
    def changeform_action_mixed_permissions_true(self, request, object_id):
        messages.success(
            request,
            "Changeform action with mixed permissions true successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changeform action with mixed permissions false",
        permissions=["example.view_user", "changeform_action_false"],
    )
    def changeform_action_mixed_permissions_false(self, request, object_id):
        messages.success(
            request,
            "Changeform action with mixed permissions false successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changeform action with mixed permissions perm not granted",
        permissions=["example.delete_user", "changeform_action_true"],
    )
    def changeform_action_mixed_permissions_perm_not_granted(self, request, object_id):
        messages.success(
            request,
            "Changeform action with mixed permissions perm not granted successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(description="Changeform action")
    def changeform_action(self, request, object_id):
        messages.success(request, "Changeform action successfully executed")
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changeform action permission true",
        permissions=["changeform_action_true"],
    )
    def changeform_action_permission_true(self, request, object_id):
        messages.success(
            request,
            "Changeform action with true permission successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changeform action permission false",
        permissions=["changeform_action_false"],
    )
    def changeform_action_permission_false(self, request, object_id):
        messages.success(
            request,
            "Changeform action with false permission successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Changeform action with multiple permissions",
        permissions=[
            "changeform_action_true",
            "changeform_action_false",
        ],
    )
    def changeform_action_multiple_different_permissions(self, request, object_id):
        messages.success(
            request,
            "Changeform action with multiple different permissions successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    def has_changeform_action_true_permission(self, request, object_id):
        return True

    def has_changeform_action_false_permission(self, request, object_id):
        return False

    ######################################################################
    # Submit line actions
    ######################################################################
    @action(description="Submit line action")
    def submit_line_action(self, request, obj):
        messages.success(request, "Submit line action successfully executed")

    @action(
        description="Submit line action with mixed permissions true",
        permissions=["submit_line_action_true", "example.view_user"],
    )
    def submit_line_action_mixed_permissions_true(self, request, obj):
        messages.success(
            request,
            "Submit line action with mixed permissions true successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Submit line action with mixed permissions false",
        permissions=["submit_line_action_false", "example.view_user"],
    )
    def submit_line_action_mixed_permissions_false(self, request, obj):
        messages.success(
            request,
            "Submit line action with mixed permissions false successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Submit line action with mixed permissions perm not granted",
        permissions=["submit_line_action_true", "example.delete_user"],
    )
    def submit_line_action_mixed_permissions_perm_not_granted(self, request, obj):
        messages.success(
            request,
            "Submit line action with mixed permissions perm not granted successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    @action(
        description="Submit line action permission true",
        permissions=["submit_line_action_true"],
    )
    def submit_line_action_permission_true(self, request, obj):
        messages.success(
            request,
            "Submit line action with true permission successfully executed",
        )

    @action(
        description="Submit line action permission false",
        permissions=["submit_line_action_false"],
    )
    def submit_line_action_permission_false(self, request, obj):
        messages.success(
            request,
            "Submit line action with false permission successfully executed",
        )

    @action(
        description="Submit line action with multiple permissions",
        permissions=[
            "submit_line_action_true",
            "submit_line_action_false",
        ],
    )
    def submit_line_action_multiple_different_permissions(self, request, obj):
        messages.success(
            request,
            "Submit line action with multiple different permissions successfully executed",
        )
        return redirect(reverse_lazy("admin:example_user_changelist"))

    def has_submit_line_action_true_permission(self, request, object_id):
        return True

    def has_submit_line_action_false_permission(self, request, object_id):
        return False


@admin.register(Group)
class GroupAdmin(BaseGroupAdmin, ModelAdmin):
    pass


@admin.register(Tag)
class TagAdmin(ModelAdmin):
    search_fields = ["name"]


@admin.register(Category)
class CategoryAdmin(ModelAdmin):
    search_fields = ["name"]


@admin.register(Label)
class LabelAdmin(ModelAdmin):
    search_fields = ["name"]


@admin.register(Project)
class ProjectAdmin(ModelAdmin):
    search_fields = ["name"]


@admin.register(Task)
class TaskAdmin(ModelAdmin):
    search_fields = ["name"]
