"""
Normalized Parallel Coordinates
-------------------------------
A `Parallel Coordinates <https://en.wikipedia.org/wiki/Parallel_coordinates>`_
chart is a chart that lets you visualize the individual data points by drawing
a single line for each of them.

Such a chart can be created in Altair by first transforming the data into a
suitable representation.

This example shows a modified parallel coordinates chart with the Penguins dataset,
where the y-axis shows the value after min-max rather than the raw value. It's a
simplified Altair version of `the VegaLite version <https://vega.github.io/vega-lite/examples/parallel_coordinate.html>`_
"""
# category: advanced calculations
import altair as alt
from altair.datasets import data
from altair import datum

source = data.penguins()

alt.Chart(source).transform_window(
    index='count()'
).transform_fold(
    ['Beak Length (mm)', 'Beak Depth (mm)', 'Flipper Length (mm)']
).transform_joinaggregate(
     min='min(value)', 
     max='max(value)',
     groupby=['key']
).transform_calculate(
    minmax_value=(datum.value-datum.min)/(datum.max-datum.min), 
    mid=(datum.min+datum.max)/2
).mark_line().encode(
    x='key:N',
    y='minmax_value:Q',
    color='Species:N',
    detail='index:N',
    opacity=alt.value(0.5)
).properties(width=500)


