//                           _       _
// __      _____  __ ___   ___  __ _| |_ ___
// \ \ /\ / / _ \/ _` \ \ / / |/ _` | __/ _ \
//  \ V  V /  __/ (_| |\ V /| | (_| | ||  __/
//   \_/\_/ \___|\__,_| \_/ |_|\__,_|\__\___|
//
//  Copyright © 2016 - 2026 Weaviate B.V. All rights reserved.
//
//  CONTACT: hello@weaviate.io
//

package tests

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/weaviate/weaviate/entities/models"
	pb "github.com/weaviate/weaviate/grpc/generated/protocol/v1"
	"github.com/weaviate/weaviate/test/helper"
	grpchelper "github.com/weaviate/weaviate/test/helper/grpc"
	"github.com/weaviate/weaviate/test/helper/sample-schema/planets"
)

func testGenerativeAWS(rest, grpc, region string) func(t *testing.T) {
	return func(t *testing.T) {
		helper.SetupClient(rest)
		helper.SetupGRPCClient(t, grpc)
		// Define path to test/helper/sample-schema/planets/data folder
		dataFolderPath := "../../../test/helper/sample-schema/planets/data"
		// Data
		data := planets.Planets
		// Define class
		class := planets.BaseClass("PlanetsGenerativeTest")
		class.VectorConfig = map[string]models.VectorConfig{
			"description": {
				Vectorizer: map[string]interface{}{
					"text2vec-aws": map[string]interface{}{
						"properties":         []interface{}{"description"},
						"vectorizeClassName": false,
						"service":            "bedrock",
						"region":             region,
						"model":              "amazon.titan-embed-text-v2:0",
					},
				},
				VectorIndexType: "flat",
			},
		}
		tests := []struct {
			name               string
			generativeModel    string
			absentModuleConfig bool
			withImages         bool
			maxTokens          int64
		}{
			// Amazon Nova
			{
				name:            "amazon.nova-micro-v1:0",
				generativeModel: "amazon.nova-micro-v1:0",
			},
			{
				name:            "amazon.nova-lite-v1:0",
				generativeModel: "amazon.nova-lite-v1:0",
				withImages:      true,
			},
			{
				name:            "amazon.nova-pro-v1:0",
				generativeModel: "amazon.nova-pro-v1:0",
				withImages:      true,
			},
			// Anthropic
			{
				name:            "anthropic.claude-3-5-sonnet-20240620-v1:0",
				generativeModel: "anthropic.claude-3-5-sonnet-20240620-v1:0",
				maxTokens:       4000,
			},
			{
				name:            "anthropic.claude-3-sonnet-20240229-v1:0",
				generativeModel: "anthropic.claude-3-sonnet-20240229-v1:0",
			},
			{
				name:            "anthropic.claude-3-haiku-20240307-v1:0",
				generativeModel: "anthropic.claude-3-haiku-20240307-v1:0",
				withImages:      true,
			},
			// Cohere
			{
				name:            "cohere.command-r-v1:0",
				generativeModel: "cohere.command-r-v1:0",
			},
			{
				name:            "cohere.command-r-plus-v1:0",
				generativeModel: "cohere.command-r-plus-v1:0",
			},
			// Meta
			{
				name:            "meta.llama3-8b-instruct-v1:0",
				generativeModel: "meta.llama3-8b-instruct-v1:0",
			},
			{
				name:            "meta.llama3-70b-instruct-v1:0",
				generativeModel: "meta.llama3-70b-instruct-v1:0",
			},
			{
				name:               "absent module config",
				generativeModel:    "meta.llama3-70b-instruct-v1:0",
				absentModuleConfig: true,
			},
			// Mistral AI
			{
				name:            "mistral.mistral-7b-instruct-v0:2",
				generativeModel: "mistral.mistral-7b-instruct-v0:2",
			},
			{
				name:            "mistral.mixtral-8x7b-instruct-v0:1",
				generativeModel: "mistral.mixtral-8x7b-instruct-v0:1",
			},
			{
				name:            "mistral.mistral-large-2402-v1:0",
				generativeModel: "mistral.mistral-large-2402-v1:0",
			},
		}
		for _, tt := range tests {
			t.Run(tt.name, func(t *testing.T) {
				if tt.absentModuleConfig {
					t.Log("skipping adding module config configuration to class")
				} else {
					class.ModuleConfig = map[string]interface{}{
						"generative-aws": map[string]interface{}{
							"service": "bedrock",
							"region":  region,
							"model":   tt.generativeModel,
						},
					}
				}
				// create schema
				helper.CreateClass(t, class)
				defer helper.DeleteClass(t, class.Class)
				// create objects
				t.Run("create objects", func(t *testing.T) {
					if tt.withImages {
						planets.InsertObjectsWithImages(t, class.Class, dataFolderPath)
					} else {
						planets.InsertObjects(t, class.Class)
					}
				})
				t.Run("check objects existence", func(t *testing.T) {
					for _, planet := range data {
						t.Run(planet.ID.String(), func(t *testing.T) {
							obj, err := helper.GetObject(t, class.Class, planet.ID, "vector")
							require.NoError(t, err)
							require.NotNil(t, obj)
							require.Len(t, obj.Vectors, 1)
							require.IsType(t, []float32{}, obj.Vectors["description"])
							assert.True(t, len(obj.Vectors["description"].([]float32)) > 0)
						})
					}
				})
				// generative task
				if tt.absentModuleConfig {
					t.Log("skipping create tweet tests with default values as e2e tests rely on specific AWS settings")
				} else {
					t.Run("create a tweet", func(t *testing.T) {
						planets.CreateTweetTest(t, class.Class)
					})
					t.Run("create a tweet using grpc", func(t *testing.T) {
						planets.CreateTweetTestGRPC(t, class.Class)
					})
				}
				t.Run("create a tweet with params", func(t *testing.T) {
					params := "aws:{temperature:0.1}"
					if tt.absentModuleConfig {
						params = fmt.Sprintf("aws:{temperature:0.1 maxTokens:500 service:\"bedrock\" region:\"%s\" model:\"%s\"}", region, tt.generativeModel)
					}
					planets.CreateTweetTestWithParams(t, class.Class, params)
				})

				params := func() *pb.GenerativeAWS {
					params := &pb.GenerativeAWS{
						Model:       grpchelper.ToPtr(tt.generativeModel),
						Temperature: grpchelper.ToPtr(0.9),
					}
					if tt.absentModuleConfig {
						params.Region = grpchelper.ToPtr(region)
						params.Service = grpchelper.ToPtr("bedrock")
					}
					if tt.maxTokens != 0 {
						params.MaxTokens = &tt.maxTokens
					}
					return params
				}

				t.Run("create a tweet with params using grpc", func(t *testing.T) {
					planets.CreateTweetTestWithParamsGRPC(t, class.Class, &pb.GenerativeProvider{
						ReturnMetadata: false, // no metadata for aws
						Kind:           &pb.GenerativeProvider_Aws{Aws: params()},
					})
				})
				if tt.withImages {
					t.Run("image prompt", func(t *testing.T) {
						t.Run("graphql", func(t *testing.T) {
							prompt := "Caption image"
							params := "aws:{imageProperties:\"image\"}"
							planets.CreatePromptTestWithParams(t, class.Class, prompt, params)
						})

						singlePrompt := "Give a short answer: What's on the image?"
						groupPrompt := "Give a short answer: What are on the following images?"

						t.Run("grpc server stored images", func(t *testing.T) {
							params := params()
							params.ImageProperties = &pb.TextArray{Values: []string{"image"}}
							planets.CreatePromptTestWithParamsGRPC(t, class.Class, singlePrompt, groupPrompt, &pb.GenerativeProvider{
								ReturnMetadata: false,
								Kind:           &pb.GenerativeProvider_Aws{Aws: params},
							})
						})

						t.Run("grpc user provided images", func(t *testing.T) {
							earth, err := planets.GetImageBlob(dataFolderPath, "earth")
							require.NoError(t, err)
							mars, err := planets.GetImageBlob(dataFolderPath, "mars")
							require.NoError(t, err)

							params := params()
							params.Images = &pb.TextArray{Values: []string{earth, mars}}
							planets.CreatePromptTestWithParamsGRPC(t, class.Class, singlePrompt, groupPrompt, &pb.GenerativeProvider{
								ReturnMetadata: false,
								Kind:           &pb.GenerativeProvider_Aws{Aws: params},
							})
						})

						t.Run("grpc mixed images", func(t *testing.T) {
							earth, err := planets.GetImageBlob(dataFolderPath, "earth")
							require.NoError(t, err)
							mars, err := planets.GetImageBlob(dataFolderPath, "mars")
							require.NoError(t, err)

							params := params()
							params.Images = &pb.TextArray{Values: []string{earth, mars}}
							params.ImageProperties = &pb.TextArray{Values: []string{"image"}}
							planets.CreatePromptTestWithParamsGRPC(t, class.Class, singlePrompt, groupPrompt, &pb.GenerativeProvider{
								ReturnMetadata: false,
								Kind:           &pb.GenerativeProvider_Aws{Aws: params},
							})
						})
					})
				}
			})
		}
	}
}
